/*
 *    WeatherCloud April 2020
 *    Developed by Plastibots.com Please keep this info intact and give credit where due. 
 *    What does it do:  Connects to darksky, gets the local weather and translates the information into LED patterns.
 *    For example, if it's raining, the led's will twinkle blue, partly cloudy will light the sun orb and show some cloud lightning
 *    Refer to www.plastibots.com "WIoT WeatherCloud" for more info.
 *     
 *    Credits:
 *      Adafruit: https://learn.adafruit.com/adafruit-neopixel-uberguide/the-magic-of-neopixels
 *      (unsure of author):  https://www.tweaking4all.com/hardware/arduino/adruino-led-strip-effects/

      Hardware:  
        Using these LED strip  https://www.banggood.com/5M-45W-150SMD-WS2812B-LED-RGB-Colorful-Strip-Light-Waterproof-IP65-WhiteBlack-PCB-DC5V-p-1035640.html?p=R90804949020201412E4
        Using this NodeMCU ESP8266:  http://www.banggood.com/Mini-NodeMCU-ESP8266-WIFI-Development-Board-Based-On-ESP-12F-p-1054209.html?p=R90804949020201412E4

 v3: stable.  Have light patterns for most weather types (e.g. rain, snow, sun, etc).  Full integration with Blynk to control cloud with other fun non-weather patterns.
 V2: beta. Weather parsing added.       
 V1: beta. Compiled version including LED features.
*/

/* Comment this out to disable prints and save space */
//#define BLYNK_PRINT Serial
#include <EEPROM.h>
#include <SimpleTimer.h>
#include <BlynkSimpleEsp8266.h>
#include <ArduinoJson.h>
#include <string.h>
#include <WiFiClientSecure.h>
#include <Adafruit_NeoPixel.h> //https://github.com/adafruit/Adafruit_NeoPixel

#define EEPROM_SIZE 4           // define the number of bytes you want to access
#define pxlCount 73             // Number of LEDs in pixel strip.
#define pixelPin D4            //LEDs pin
#define VISUALS 6

//############################################################################################################################
//F O R    D E B U G G I N G 
boolean debug = true;   //Be sure to set to false when done debugging!!
//############################################################################################################################

//############################################################################################################################
//D A T A     P R O V I D E R
//DarkSky.net   *** if you dont have a DarkSky API, thank Apple for it no longer being free as of Spring 2020. Try out https://www.weatherbit.io.  You will have to modify this code.
//Docs:  https://darksky.net/dev/docs
//API Call Sample:  https://api.darksky.net/forecast/[APIKEYHERE]/43.3989896,-79.78224415?exclude=minutely,hourly
//To get coordinates, open Google Maps. Right-click the place or area on the map.Select What's here? At the bottom, you’ll see a card with the coordinates.

//DarkSky API Connection Data - uncomment this section if using
const char Host[] = "api.darksky.net";
const int httpsPort = 443;
//const char* fingerprint = "eb c2 67 d1 b1 c6 77 90 51 c1 4a 0a ba 83 e1 f0 6d 73 dd b8";// Use web browser to view and copy // SHA1 fingerprint of the certificate
const char apiKey[] = "DarkSky API Key Here";   //API Key
//Locations:  Burlington: 43.3989896,-79.78224415   Hamilton: same
char wLocation[] = "43.3249,-79.7967"; //format=Lat,Long  Hamilton ON - default.  Changes based on Nextion location screen
char weatherRecLoc[30] = "Burlington";
const int httpPort = 80;
const int tzEpochCorrect = 14400;  // Seconds for GMT EST -4 hours (includes DST)
//############################################################################################################################
B L Y N K

//Blynk connection credentials
char auth[] = "Blynk Auth";
char ssid[] = "SSID";
char pass[] = "Password";
//############################################################################################################################

int LEDpattern = 1;  //set default
int lastLEDpattern = 0;
int realColor = 0;
int LEDbrightness = 0;

//int reading;
int previous = LOW;

long times = 0;
long debounce = 200;

uint32_t redValue = 0;
uint32_t greenValue = 0;
uint32_t blueValue = 0;
uint32_t redcheer = 0;
uint32_t greencheer = 0;
uint32_t bluecheer = 0;

uint16_t gradient = 0;
uint16_t thresholds[] = {1529, 1019, 764, 764, 764, 1274};
uint8_t palette = 0;
uint8_t visual = 0;
uint8_t last = 0;
bool shuffle = true;
bool bump = true;
bool left = false;
int8_t dotPos = 0;
float timeBump = 0;
float avgBump = 0;
float avgTime = 0;
float shuffleTime = 0;
int8_t pos[pxlCount] = {-2};
uint8_t rgb[pxlCount][3] = {0};

//Weather vars
int weatherType = 0;
float cloudCover = 0.00;  //provides indicator in % of cloud cover.  100% is no sun.
int uvIndex = 0;

int tmrClearMsg; 
int msgShowDuration = 8000;

const long refreshMS = 60*60*1000;   // Min * Sec * MilliSec    900000 = 15 mins | 300000 - 5 mins | 60000 - 1 minute | 3600000 - 1 hr before next update.
unsigned long prevRefreshMS = 0;  

boolean firsttime = true;

//LEDs
Adafruit_NeoPixel strip = Adafruit_NeoPixel(pxlCount, pixelPin, NEO_GRB + NEO_KHZ800);
SimpleTimer timer;                                    
WiFiClientSecure secureWiFiClient;  // Instance of secure client to enable HTTPS request


BLYNK_WRITE(V0)  //All LEDs off
{
  LEDpattern = 0;
  setAll(0, 0, 0);
  Blynk.virtualWrite(V5, 0);
  Blynk.virtualWrite(V6, 0);
  Blynk.virtualWrite(V1, "LEDs Off"); 
  timer.restartTimer(tmrClearMsg);
}

//######################################################################################
BLYNK_WRITE(V5)  //Get Segmented Switch #1 Option from Blynk
{
 Blynk.virtualWrite(V6, 0);  //clear 2nd row option selection
 switch (param.asInt())
  {
    case 1: {
        LEDpattern = 1;
        break;
      }
    case 2: { // RGB Loop
        LEDpattern = 2;
        break;
      }
    case 3: { // Cylon
        LEDpattern = 3;
        break;
      }
    case 4: { // Twinkle
        LEDpattern = 4;
        break;
      }
    case 5: { // Twinkle Random
        LEDpattern = 5;
        break;
      }
    }

    lastLEDpattern = LEDpattern;
    Blynk.virtualWrite(V1, "Pattern Changed:" + String(LEDpattern)); 
    timer.restartTimer(tmrClearMsg);

}
//######################################################################################

BLYNK_WRITE(V6)  //Get Segmented Switch #2 Option from Blynk
{
 Blynk.virtualWrite(V5, 0);
 switch (param.asInt())
  {
    case 1: { // Sparkle
        LEDpattern = 6;
        break;
      }
    case 2: { // Snow Sprinkle
        LEDpattern = 7;
        break;
      }
    case 3: { // Colour Wipe
        LEDpattern = 8;
        break;
      }
    case 4: { // Rainbow
        LEDpattern = 9;
        break;
      }
    case 5: { // Bouncing Balls
        LEDpattern = 10;
        break;
      }
    }

    lastLEDpattern = LEDpattern;
    Blynk.virtualWrite(V1, "Pattern Changed:" + String(LEDpattern)); 
    timer.restartTimer(tmrClearMsg); 
}     

        

//######################################################################################
BLYNK_WRITE(V4) // zeRGBa assigned to V4
{
    setAll(0, 0, 0); //clear all LEDs
    // get a RED channel value
    redValue = param[0].asInt();
    // get a GREEN channel value
     greenValue = param[1].asInt();
    // get a BLUE channel value
    blueValue = param[2].asInt();
    Blynk.virtualWrite(V1, "Colour changed.");   
    LEDpattern = 11;  //for RGB
    lastLEDpattern = LEDpattern;  
    Blynk.virtualWrite(V5, 0);
    Blynk.virtualWrite(V6, 0);
    timer.restartTimer(tmrClearMsg);
}

//######################################################################################
BLYNK_WRITE(V7)  //LED Brightness Slider
{
  int val = param.asInt();
  EEPROM.put(0, val);  //write val to EEPROM byte # 0.  Note, no need to check if different as the put function only writes if the value is different.
  EEPROM.commit();
  LEDbrightness = val;
  strip.setBrightness(LEDbrightness);
  delay(200);
  
  Blynk.virtualWrite(V1, "Brightness updated:" + String(val));
  timer.restartTimer(tmrClearMsg); 
}

//######################################################################################
BLYNK_WRITE(V8) // Music VU Mode LED Patterns
{
  LEDpattern = 1;  //set so that actually weather doesnt take over.
  lastLEDpattern = LEDpattern;
  setAll(0,0,0);
  Blynk.virtualWrite(V5, 0);
  Blynk.virtualWrite(V6, 0);
  switch (param.asInt())  //allows changing the colour patterns
  {
    case 1: Blynk.virtualWrite(V1, "P-Cloudy Day"); weatherType=1; break;
    case 2: Blynk.virtualWrite(V1, "P-Cloudy Night"); weatherType=2; break;
    case 3: Blynk.virtualWrite(V1, "Clear Day"); weatherType=3; break;
    case 4: Blynk.virtualWrite(V1, "Cloudy"); weatherType=4; break;
    case 5: Blynk.virtualWrite(V1, "Snow"); weatherType=5; break;
    case 6: Blynk.virtualWrite(V1, "Fog"); weatherType=6; break;
    case 7: Blynk.virtualWrite(V1, "Hazy"); weatherType=7; break;
    case 8: Blynk.virtualWrite(V1, "Rain"); weatherType=8; break;
    case 9: Blynk.virtualWrite(V1, "Sleet"); weatherType=9; break;
    case 10: Blynk.virtualWrite(V1, "T-Storm"); weatherType=10; break;
    case 11: Blynk.virtualWrite(V1, "Tornado WTF?!"); weatherType=11; break;
    case 12: Blynk.virtualWrite(V1, "Hail"); weatherType=12; break;
    break;
  }       
}
//######################################################################################
void clearBlynkStatMsg()
{
    Blynk.virtualWrite(V1, " ");
    //timer.disable(tmrClearMsg);
}
//######################################################################################


void setup() {
  pinMode(pixelPin, OUTPUT);
  
  Serial.begin(115200);
 
  EEPROM.begin(EEPROM_SIZE); // initialize EEPROM with predefined size
  LEDbrightness = EEPROM.read(0);  //get the stored brightness level.
  delay(50);
  strip.begin();
  delay(200);
  strip.setBrightness(LEDbrightness);
  setAll(0,0,0);
  
  WiFi.hostname("WIotWeatherCloud");  //set the name as friendly - will show in Router device list.
  
  Blynk.begin(auth, ssid, pass);  
  if (debug) {Serial.print(F("Connecting to Blynk"));}
  while (Blynk.connect() == false) {
     //Wait until connected
     if (debug) {Serial.println(".");}
  }
  if (debug) {Serial.println(F("WiFi Connected!"));}
  if (debug) {Serial.println(F(""));}
 
  Blynk.virtualWrite(V1, " ");
  Blynk.virtualWrite(V5, 0);
  Blynk.virtualWrite(V6, 0);
  Blynk.virtualWrite(V8, 0);  //music mode
  Blynk.virtualWrite(V2, ""); 
  Blynk.virtualWrite(V2, WiFi.localIP().toString()); 
  

  tmrClearMsg = timer.setInterval(msgShowDuration, clearBlynkStatMsg);  //timed delay for shutting off LEDs after they have come on.
  

}
//######################################################################################

void loop() 
{
  
  Blynk.run();  //must  be here. Avoid any delay()'s.
  timer.run(); 
 
  unsigned long currentMillis = millis();    //for sleep feature

  if (firsttime && currentMillis > 10000) //after a break, get data on startup.
  {
    updateWeather();
    firsttime = false;
  }
  else if (currentMillis - prevRefreshMS > refreshMS)
  {
    prevRefreshMS = currentMillis; // save the last time we checked
    updateWeather();    
  }

  /* 
  if (debug) {Serial.print("  Pattern: ");Serial.print(LEDpattern);}
  if (debug) {Serial.print("  Millis: ");Serial.println(millis());}
  */

 // THIS IS FOR WEATHER AND OTHER LED PATTERNS. 
 switch (LEDpattern)
  {
    case 1: displayWeatherCloud(weatherType); break;  //FOR ALL WEATHER ITEMS
    case 2: RGBLoop(LEDpattern);  break;
    case 3: CylonBounce(random(0, 255), random(0, 255), random(0, 255), 4, 50, 100, LEDpattern);  break;
    case 4: Twinkle(0xE3, 0x20, 0xD6, 20, 100, false, LEDpattern);  break;
    case 5: TwinkleRandom(6, 100, false, LEDpattern);  break;
    case 6: Sparkle(random(255), random(255), random(255), 100);  break;
    case 7: SnowSparkle(16, 16, 16, 20, 100, LEDpattern);  break;
    case 8: colorWipe(random(0, 255), random(0, 255), random(0, 255), 50, LEDpattern);  break;
    case 9: theaterChaseRainbow(100, LEDpattern);  break;
    case 10: BouncingBalls(0, 0, 255, 26, LEDpattern);  break;
    case 11: setLed();  break;
   }
}
//######################################################################################
bool updateWeather()
{
  char conStr[200];
  char status[32] = {0};
  char endOfHeaders[] = "\r\n\r\n";
  char comma[] = ",";
  char endOfArray[] = "]";


  //https://api.darksky.net/forecast/[APIKEY]/43.3989896,-79.78224415?exclude=minutely,daily,hourly,flags
 
  strcpy(conStr, "GET /forecast/");
  strcat(conStr, apiKey);
  strcat(conStr, "/");
  strcat(conStr, wLocation);
  strcat(conStr, "?units=ca&exclude=minutely,hourly,daily,alerts,flags");
  strcat(conStr, " HTTP/1.1\r\nUser-Agent: ESP8266/0.1\r\nAccept: */*\r\nHost: ");
  strcat(conStr, Host);
  strcat(conStr, "\r\nConnection: close\r\n\r\n");  

  if (debug) {Serial.println(F("Connect to host.."));Serial.println(conStr);}

  secureWiFiClient.setInsecure();
  if (!secureWiFiClient.connect(Host, 443)) {
      if (debug) {Serial.println(F("Get Curr Weather Data: Connection failed.."));}
      return false;
  }

  yield();
  
  //SEND THE REQUEST
  if (debug) {Serial.println(conStr);}
  if (debug) {Serial.println(F("Get Curr Weather Data: Send the connection string.."));}
  secureWiFiClient.print(conStr);
  yield();
  delay(1000);


  //use this to dump response.  
  //Serial.print("Response Header: ");
  //while(secureWiFiClient.available()){
  //  char stat[1];
  //  secureWiFiClient.readBytesUntil('\r', stat, sizeof(stat));
  //     //secureWiFiClient.readBytes(stat, sizeof(stat));
  //  Serial.print(stat);
  //}
  //Serial.println();

  // Check HTTP status
  if (debug) {Serial.println(F("Get Curr Weather Data: Check HTTP return status.."));}
  secureWiFiClient.readBytesUntil('\r', status, sizeof(status));
  yield();
    if (strcmp(status, "HTTP/1.1 200 OK") != 0) {
    if (debug) {Serial.print(F("Current Data: Unexpected response. Status received = "));}
    if (debug) {Serial.println(status);}
    return false;
  }
  yield();
  // Skip HTTP headers
  if (!secureWiFiClient.find(endOfHeaders)) {
    if (debug) {Serial.println(F("Invalid response"));}
    return false;
  }

  DynamicJsonDocument jsonDoc(1024);  //small payload
  DeserializationError error = deserializeJson(jsonDoc, secureWiFiClient);
  if (error)
  {
    if (debug) {Serial.print(F("jsonDocCurrWeather failed with code "));}
    if (debug) {Serial.println(error.c_str());}
    return false;
  }

  if (debug) {Serial.print(F("jsonDoc Size: "));Serial.println(jsonDoc.size());}

  secureWiFiClient.stop();  //done with connection
  if (debug) {Serial.print(F(" Location ")); Serial.println(weatherRecLoc);}

  weatherType = iconNameToInt(jsonDoc["currently"]["icon"].as<char*>());
  if (debug) {Serial.print(F("IconName: "));Serial.print(jsonDoc["currently"]["icon"].as<char*>());Serial.print(F(" >> Translated="));Serial.print(weatherType);}
  cloudCover = jsonDoc["currently"]["cloudCover"].as<float>();
  if (debug) {Serial.print(F("  |  CloudCover (float)%: "));Serial.print(cloudCover);}
  uvIndex = jsonDoc["currently"]["uvIndex"];
  if (debug) {Serial.print(F("  |  UvIndex (int): "));Serial.println(uvIndex);}

  //data will be used for LEDPattern named weathercloud
  displayWeatherCloud(weatherType);  //update weather.

  return true;
}
//######################################################################################
void displayWeatherCloud(int wt)
{
  //Sse weatherType (rain, sunny etc as ints) + cloudCover (to determine % of clouds vs sun) + uVIndex (intensity of sun)
  //weatherType, cloudCover(float), uvIndex (0-10)
  //per iconNameToInt() copied here for ease of use: partly-cloudy-day = 1   partly-cloudy-night = 2
  //clear-day=3  cloudy=4  snow=5  fog=6  hazy=7  rain=8 sleet=9 thunderstorm=10 tornado=11
  //hail=12 default=13

  switch (wt) 
  {
    case 1:  cloudyDay(16, 16, 16, true, 50, 200, LEDpattern); break;     //partly cloudy day - show the sun
    case 2:  pCloudyNight(5, 5, 5, 50, 200, LEDpattern); break;           //partly cloudy night
    case 3:  clearDay(LEDpattern); break;                                 // clear day
    case 4:  cloudyDay(16, 16, 16, false, 50, 200, LEDpattern); break;    //cloudy
    case 5:  SnowSparkle(16, 16, 16, 20, 100, LEDpattern); break;         //snow
    case 6:  cloudyDay(5, 5, 5, false, 50, 200, LEDpattern); break;       //fog
    case 7:  hazy(LEDpattern);break;                                      //hazy
    case 8:  rain(0, 0, 40, 150, 20, LEDpattern); break;                  //rain
    //void rain(red, green, blue, SparkleDelay, SpeedDelay, int LP) {
    case 9:  BouncingBalls(0, 0, 255, 26, LEDpattern); break;             //sleet
    case 10: Strobe(160, 160, 160, random(1, 4), random(20, 100), random(200, 3000), LEDpattern); break;  //thunderstorm
    case 11: break;  //tornado
    case 12: rain(255, 0, 0, 100, 20, LEDpattern); break;                   //hail  - light all red and flicker!
    default:  break;  //default
  }
}

//######################################################################################
int iconNameToInt(String iconcode)
{

    //returns a code that can be used to set the LED pixels
    if (iconcode.indexOf("partly-cloudy-day") >= 0)
      return 1;
    else if (iconcode.indexOf("partly-cloudy-night") >= 0)
      return 2;
    else if (iconcode.indexOf("clear-day") >= 0)
      return 3;
    else if (iconcode.indexOf("cloudy") >= 0)
      return 4;
    else if (iconcode.indexOf("snow") >= 0)
      return 5;
    else if (iconcode.indexOf("fog") >= 0)
      return 6;
    else if (iconcode.indexOf("hazy") >= 0)
      return 7;
    else if (iconcode.indexOf("rain") >= 0)
      return 8;
    else if (iconcode.indexOf("sleet") >= 0)
      return 9;
    else if (iconcode.indexOf("thunderstorm") >= 0)
      return 10;
    else if (iconcode.indexOf("tornado") >= 0)
      return 11;
    else if (iconcode.indexOf("hail") >= 0)
      return 12;
    else //unknown
      return 13;
}

//######################################################################################
//######################################################################################

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void clearDay(int LP) {
  //set sun specific
    for (int j = 28; j <= 35; j++) {
      setPixel(j, 255, 255, 255); //specific pixels set for sun colour
    }
    strip.show();
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

void hazy(int LP) {
  //set sun specific- with haze
    for (int j = 28; j <= 35; j++) {
      setPixel(j, 55, 10, 0); //specific pixels set for sun colour - hazy day
    }
    strip.show();
  
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void pCloudyNight(byte red, byte green, byte blue, int glowDelay, int SpeedDelay, int LP) {

    setAll(red, green, blue);
    //Set the moonlight
    for (int j = 70; j <= 73; j++) {
      setPixel(j, 20, 20, 80); //Set the moon to be a bit brighter and more blue
    }
    //set the sun off
    for (int j = 28; j <= 35; j++) {
      setPixel(j, 0, 0, 0); //Set the moon to be a bit brighter and more blue
    }
    strip.show(); 
 
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void cloudyDay(byte red, byte green, byte blue, bool sun, int glowDelay, int SpeedDelay, int LP) {
  //setAll(red, green, blue);
 
  for (int j = 0; j <= 69; j++) {
    setPixel(j, red, green, blue); //Set the moon to be a bit brighter and more blue
  }  
  strip.show();
  //set the sun specific pixelstrip.show();
  if (sun)
  {
    for (int j = 28; j <= 35; j++) {
      setPixel(j, 63, 46, 0); //specific pixels set for sun colour - very dim since it's cloudy
    }
    strip.show();
  }  
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void rain(byte red, byte green, byte blue, int SparkleDelay, int SpeedDelay, int LP) {
  //setAll(red, green, blue);
  for (int j=0; j<= 26; j++) {
      setPixel(j, red, green, blue); //set first batch up to sun area
    }
  for (int j=36; j<=68; j++) {
    setPixel(j, red, green, blue); //set rest after sun, but before moon (end)
  }  

  //int Pixel = random(pxlCount);
  int Pixel = random(0,68);  //random pixels for all areas except sun and moon
  if (((Pixel >= 0) && (Pixel <= 26)) ||((Pixel >= 36) && (Pixel <= 68)))  //no sun and moon pixels
  {
    //setPixel(Pixel, 0xff, 0xff, 0xff); //255,255,255  
    setPixel(Pixel, 0, 0, 255); //blue  - sparkle pixel
  }
  
  strip.show();
  delay(SparkleDelay);
  //  setPixel(Pixel, red, green, blue);
  if (((Pixel >= 0) && (Pixel <= 26)) ||((Pixel >= 36) && (Pixel <= 68)))  //no sun and moon pixels
  {
    setPixel(Pixel, red, green, blue);  //set back to original colour
  }

  strip.show();
  delay(SpeedDelay);
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Strobe(byte red, byte green, byte blue, int StrobeCount, int FlashDelay, int EndPause, int LP) {
  for (int j = 0; j < StrobeCount; j++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    //setAll(red, green, blue);
    for (int j = 0; j <= 27; j++) {
      setPixel(j, red, green, blue); //set first batch up to sun area
    }
    for (int j = 36; j <=69; j++) {
      setPixel(j, red, green, blue); //set rest after sun, but before moon (end)
    }
        
    strip.show();
    delay(FlashDelay);
    setAll(0, 0, 0);
    strip.show();
    delay(FlashDelay);
  }
  delay(1000);
  
  flashEachCloud(red, green, blue, random(1,4), StrobeCount, FlashDelay);

  delay(EndPause);
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////

void flashEachCloud(byte r, byte g, byte b, int which, int strbCount, int fdelay)
{
  int strt, fin;
  strip.setBrightness(random(200,255));
  switch (which) {
        case 1: strt=1, fin=27; break;   //start to before sun
        case 2: strt=38, fin=49; break;  //after sun up to moon
        case 3: strt=49, fin=69; break;
        case 4: strt=0, fin=0; break;
      }
  if (debug) {Serial.print(F("WhichRandomVal:"));Serial.println(which);} 
    
  for (int j = strt; j <= fin; j++) {
    setPixel(j, r, g, b);
  }
  for (int j = 0; j < strbCount; j++) {
    strip.show();
    delay(fdelay);
    setAll(0, 0, 0);
    strip.show();
    delay(fdelay);
  }
  strip.setBrightness(LEDbrightness);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void CylonBounce(byte red, byte green, byte blue, int EyeSize, int SpeedDelay, int ReturnDelay, int LP) 
{
 
  for (int i = 0; i < pxlCount - EyeSize - 2; i++) 
  {

    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed

    
    setAll(0, 0, 0);
    setPixel(i, red / 10, green / 10, blue / 10);
    for (int j = 1; j <= EyeSize; j++) 
    {
      setPixel(i + j, red, green, blue);
    }
    setPixel(i + EyeSize + 1, red / 10, green / 10, blue / 10);
    strip.show();
    delay(SpeedDelay);

    if ( i % 10 == 0) 
    {
      if (debug) {Serial.println(F("CylonBounceWorking"));} Blynk.run();
 
    } //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }

  delay(ReturnDelay);

  for (int i = pxlCount - EyeSize - 2; i > 0; i--) 
  {
      if (LP != lastLEDpattern)
      {
        if (debug) {Serial.println(F("Broken out of Cylon Bounce"));}
        break;
      }
    
    setAll(0, 0, 0);
    setPixel(i, red / 10, green / 10, blue / 10);
    for (int j = 1; j <= EyeSize; j++) {
      setPixel(i + j, red, green, blue);
    }
    setPixel(i + EyeSize + 1, red / 10, green / 10, blue / 10);
    strip.show();
    delay(SpeedDelay);
    if ( i % 10 == 0) {if (debug) {Serial.println(F("CylonBounceWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }

  delay(ReturnDelay);
}




//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void LeftToRight(byte red, byte green, byte blue, int EyeSize, int SpeedDelay, int ReturnDelay, int LP) {
  for (int i = 0; i < pxlCount - EyeSize - 2; i++) {
    setAll(0, 0, 0);
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    setPixel(i, red / 10, green / 10, blue / 10);
    for (int j = 1; j <= EyeSize; j++) {
      setPixel(i + j, red, green, blue);
    }
    setPixel(i + EyeSize + 1, red / 10, green / 10, blue / 10);
    strip.show();
    delay(SpeedDelay);
    if ( i % 50 == 0) {if (debug) {Serial.println(F("LeftToRightWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
  delay(ReturnDelay);
  
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void RightToLeft(byte red, byte green, byte blue, int EyeSize, int SpeedDelay, int ReturnDelay, int LP) {
  for (int i = pxlCount - EyeSize - 2; i > 0; i--) {
    setAll(0, 0, 0);
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    setPixel(i, red / 10, green / 10, blue / 10);
    for (int j = 1; j <= EyeSize; j++) {
      setPixel(i + j, red, green, blue);
    }
    setPixel(i + EyeSize + 1, red / 10, green / 10, blue / 10);
    strip.show();
    delay(SpeedDelay);
    if ( i % 50 == 0) {if (debug) {Serial.println(F("RightToLeftWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
  delay(ReturnDelay);
  
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Twinkle(byte red, byte green, byte blue, int Count, int SpeedDelay, boolean OnlyOne, int LP) {
  setAll(0, 0, 0);

  for (int i = 0; i < Count; i++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    setPixel(random(pxlCount), red, green, blue);
    strip.show();
    delay(SpeedDelay);
    if (OnlyOne) {
      setAll(0, 0, 0);
    }
    if ( i % 50 == 0) {if (debug) {Serial.println(F("TwinkleWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }

  delay(SpeedDelay);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void TwinkleRandom(int Count, int SpeedDelay, boolean OnlyOne, int LP) {
  setAll(0, 0, 0);
  
  for (int i = 0; i < Count; i++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    setPixel(random(pxlCount), random(0, 255), random(0, 255), random(0, 255));
    strip.show();
    delay(SpeedDelay);
    if (OnlyOne) {
      setAll(0, 0, 0);
    }
    if ( i % 50 == 0) {if (debug) {Serial.println(F("TwinkleRandomWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }

  delay(SpeedDelay);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Sparkle(byte red, byte green, byte blue, int SpeedDelay) {
  int Pixel = random(pxlCount);
  setPixel(Pixel, red, green, blue);
  strip.show();
  delay(SpeedDelay);
  setPixel(Pixel, 0, 0, 0);
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void SnowSparkle(byte red, byte green, byte blue, int SparkleDelay, int SpeedDelay, int LP) {
  setAll(red, green, blue);

  int Pixel = random(pxlCount);
  setPixel(Pixel, 0xff, 0xff, 0xff);
  strip.show();
  delay(SparkleDelay);
  setPixel(Pixel, red, green, blue);
  strip.show();
  delay(SpeedDelay);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void RunningLights(byte red, byte green, byte blue, int WaveDelay, int LP) {
  int Position = 0;

  for (int i = 0; i < pxlCount * 2; i++)
  {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    Position++; // = 0; //Position + Rate;
    for (int i = 0; i < pxlCount; i++) {
      setPixel(i, ((sin(i + Position) * 127 + 128) / 255)*red,
               ((sin(i + Position) * 127 + 128) / 255)*green,
               ((sin(i + Position) * 127 + 128) / 255)*blue);
    }

    strip.show();
    delay(WaveDelay);
    if ( i % 40 == 0) {if (debug) {Serial.println(F("RunningLightsWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void colorWipe(byte red, byte green, byte blue, int SpeedDelay, int LP) {
  for (uint16_t i = 0; i < pxlCount; i++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    setPixel(i, red, green, blue);
    strip.show();
    delay(SpeedDelay);
    if ( i % 50 == 0) {if (debug) {Serial.println(F("ColourWipeWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }

}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void rainbowCycle(int SpeedDelay, int LP) {
  byte *c;
  uint16_t i, j;

  for (j = 0; j < 256*5; j++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    for (i = 0; i < pxlCount; i++) {
      if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
      c = Wheel(((i * 256 / pxlCount) + j) & 255);
      setPixel(i, *c, *(c + 1), *(c + 2));
    }
    strip.show();
    delay(SpeedDelay);
    if ( j % 50 == 0) {if (debug) {Serial.println(F("RainbowCycleWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void theaterChase(byte red, byte green, byte blue, int SpeedDelay, int LP) {
  for (int j = 0; j < 10; j++) { //do 10 cycles of chasing
    for (int q = 0; q < 3; q++) {
      if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
      for (int i = 0; i < pxlCount; i = i + 3) {
        if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
        setPixel(i + q, red, green, blue);
      }
      strip.show();

      delay(SpeedDelay);

      for (int i = 0; i < pxlCount; i = i + 3) {
        setPixel(i + q, 0, 0, 0);
      }
    }
  if ( j % 2 == 0) {if (debug) {Serial.println(F("TheatreChaseWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void theaterChaseRainbow(int SpeedDelay, int LP) {
  byte *c;

  for (int j = 0; j < 256; j++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    for (int q = 0; q < 3; q++) {
      if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
      for (int i = 0; i < pxlCount; i = i + 3) {
        if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
        c = Wheel( (i + j) % 255);
        setPixel(i + q, *c, *(c + 1), *(c + 2));
      }
      strip.show();

      delay(SpeedDelay);

      for (int i = 0; i < pxlCount; i = i + 3) {
        if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
        setPixel(i + q, 0, 0, 0);
      }
    }
    if ( j % 50 == 0) {if (debug) {Serial.println(F("TheatreChaseRaimbowWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Fire(int Cooling, int Sparking, int SpeedDelay, int LP) {
  static byte heat[pxlCount];
  int cooldown;

  // Step 1.  Cool down every cell a little
  for ( int i = 0; i < pxlCount; i++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    cooldown = random(0, ((Cooling * 10) / pxlCount) + 2);

    if (cooldown > heat[i]) {
      heat[i] = 0;
    } else {
      heat[i] = heat[i] - cooldown;
    }
    if ( i % 50 == 0) {if (debug) {Serial.println(F("FireWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }

  // Step 2.  Heat from each cell drifts 'up' and diffuses a little
  for ( int k = pxlCount - 1; k >= 2; k--) {
    heat[k] = (heat[k - 1] + heat[k - 2] + heat[k - 2]) / 3;
  }

  // Step 3.  Randomly ignite new 'sparks' near the bottom
  if ( random(255) < Sparking ) {
    int y = random(7);
    heat[y] = heat[y] + random(160, 255);
    //heat[y] = random(160,255);
  }

  // Step 4.  Convert heat to LED colors
  for ( int j = 0; j < pxlCount; j++) {
    setPixelHeatColor(j, heat[j] );
  }

  strip.show();
  delay(SpeedDelay);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setPixelHeatColor (int Pixel, byte temperature) {
  // Scale 'heat' down from 0-255 to 0-191
  byte t192 = round((temperature / 255.0) * 191);

  // calculate ramp up from
  byte heatramp = t192 & 0x3F; // 0..63
  heatramp <<= 2; // scale up to 0..252

  // figure out which third of the spectrum we're in:
  if ( t192 > 0x80) {                    // hottest
    setPixel(Pixel, 255, 255, heatramp);
  } else if ( t192 > 0x40 ) {            // middle
    setPixel(Pixel, 255, heatramp, 0);
  } else {                               // coolest
    setPixel(Pixel, heatramp, 0, 0);
  }
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void BouncingBalls(byte red, byte green, byte blue, int BallCount, int LP) {

  if (debug) {Serial.println(F("BouncingBalls Started"));}
  float Gravity = -9.81;
  int StartHeight = 1;

  float Height[BallCount];
  float ImpactVelocityStart = sqrt( -2 * Gravity * StartHeight );
  float ImpactVelocity[BallCount];
  float TimeSinceLastBounce[BallCount];
  int   Position[BallCount];
  long  ClockTimeSinceLastBounce[BallCount];
  float Dampening[BallCount];
  int whilecounter=0;

  for (int i = 0 ; i < BallCount ; i++) {
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    ClockTimeSinceLastBounce[i] = millis();
    Height[i] = StartHeight;
    Position[i] = 0;
    ImpactVelocity[i] = ImpactVelocityStart;
    TimeSinceLastBounce[i] = 0;
    Dampening[i] = 0.90 - float(i) / pow(BallCount, 2);
  }

 while (whilecounter<100) {
    whilecounter++;
    if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    for (int i = 0 ; i < BallCount ; i++) {
      if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
      TimeSinceLastBounce[i] =  millis() - ClockTimeSinceLastBounce[i];
      Height[i] = 0.5 * Gravity * pow( TimeSinceLastBounce[i] / 1000 , 2.0 ) + ImpactVelocity[i] * TimeSinceLastBounce[i] / 1000;

      if ( Height[i] < 0 ) {
        Height[i] = 0;
        ImpactVelocity[i] = Dampening[i] * ImpactVelocity[i];
        ClockTimeSinceLastBounce[i] = millis();

        if ( ImpactVelocity[i] < 0.01 ) {
          ImpactVelocity[i] = ImpactVelocityStart;
        }
      }
      Position[i] = round( Height[i] * (pxlCount - 1) / StartHeight);
    }

    for (int i = 0 ; i < BallCount ; i++) {
      setPixel(Position[i], red, green, blue);
      if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed
    }

    strip.show();
    setAll(0, 0, 0);
    delay(100);
  }
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void BouncingColoredBalls(int BallCount, byte colors[][3]) {
  float Gravity = -9.81;
  int StartHeight = 1;

  float Height[BallCount];
  float ImpactVelocityStart = sqrt( -2 * Gravity * StartHeight );
  float ImpactVelocity[BallCount];
  float TimeSinceLastBounce[BallCount];
  int   Position[BallCount];
  long  ClockTimeSinceLastBounce[BallCount];
  float Dampening[BallCount];
  int whilecounter=0;

  for (int i = 0 ; i < BallCount ; i++) {
    ClockTimeSinceLastBounce[i] = millis();
    Height[i] = StartHeight;
    Position[i] = 0;
    ImpactVelocity[i] = ImpactVelocityStart;
    TimeSinceLastBounce[i] = 0;
    Dampening[i] = 0.90 - float(i) / pow(BallCount, 2);
  }
  
 while (whilecounter<100) {
    whilecounter++;
    for (int i = 0 ; i < BallCount ; i++) {
      TimeSinceLastBounce[i] =  millis() - ClockTimeSinceLastBounce[i];
      Height[i] = 0.5 * Gravity * pow( TimeSinceLastBounce[i] / 1000 , 2.0 ) + ImpactVelocity[i] * TimeSinceLastBounce[i] / 1000;

      if ( Height[i] < 0 ) {
        Height[i] = 0;
        ImpactVelocity[i] = Dampening[i] * ImpactVelocity[i];
        ClockTimeSinceLastBounce[i] = millis();

        if ( ImpactVelocity[i] < 0.01 ) {
          ImpactVelocity[i] = ImpactVelocityStart;
        }
      }
      Position[i] = round( Height[i] * (pxlCount - 1) / StartHeight);
    }

    for (int i = 0 ; i < BallCount ; i++) {
      setPixel(Position[i], colors[i][0], colors[i][1], colors[i][2]);
    }
    strip.show();
    setAll(0, 0, 0);
    delay(100);
  }
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setPixel(int Pixel, byte red, byte green, byte blue) 
{
  strip.setPixelColor(Pixel, strip.Color(red, green, blue));
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setAll(byte red, byte green, byte blue) 
{
 
  for (int i = 0; i < pxlCount; i++ ) 
  {
    strip.setPixelColor(i, strip.Color(red, green, blue));
  }
  strip.show();     
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void getReal() {
if (realColor == 1) {                //Candle
      redValue = 125;
      greenValue = 47;
      blueValue = 0;
}
else if (realColor == 2) {                 //40W Tungsten
      redValue = 125;
      greenValue = 97;
      blueValue = 43;
}
else if (realColor == 3) {                 //100 Tungsten
      redValue = 105;
      greenValue = 64;
      blueValue = 20;
}
else if (realColor == 4) {                //Halogen
      redValue = 105;
      greenValue = 91;
      blueValue = 74;
}
else if (realColor == 5) {                 //Carbon Arc 
      redValue = 105;
      greenValue = 100;
      blueValue = 94;
}
else if (realColor == 6) {                 //High Noon Sun
      redValue = 105;
      greenValue = 105;
      blueValue = 101;
}
 else if (realColor == 7) {                 //Direct Sunlight
      redValue = 105;
      greenValue = 105;
      blueValue = 105;
}
else if (realColor == 8) {                 //Overcast Sky;
      redValue = 51;
      greenValue = 76;
      blueValue = 105;
}

else if (realColor == 9) {                 //Clear Blue Sky
      redValue = 0;
      greenValue = 56;
      blueValue = 155;
}
else if (realColor == 10) {                //Warm Fluorescent
      redValue = 105;
      greenValue = 94;
      blueValue = 79;
}
else if (realColor == 11) {                //Standard Fluorescent
      redValue = 94;
      greenValue = 105;
      blueValue = 100;
}
else if (realColor == 12) {                //Cool White Fluorescent
      redValue = 62;
      greenValue = 85;
      blueValue = 105;
}
else if (realColor == 13) {                //Full Spectrum Fluorescent
      redValue = 105;
      greenValue = 94;
      blueValue = 92;
}
else if (realColor == 14) {                //Grow Light Fluorescent
      redValue = 105;
      greenValue = 89;
      blueValue = 97;
}
else if (realColor == 15) {                //Black Light Fluorescent
      redValue = 37;
      greenValue = 0;
      blueValue = 105;
}
else if (realColor == 16) {                //Mercury Vapor
      redValue = 66;
      greenValue = 97;
      blueValue = 105;
}
else if (realColor == 17) {                //Sodium Vapor
      redValue = 105;
      greenValue = 59;
      blueValue = 28;
}
else if (realColor == 18) {                //Metal Halide
      redValue = 92;
      greenValue = 102;
      blueValue = 105;
}
else if (realColor == 19) {                //High Pressure Sodium
      redValue = 105;
      greenValue = 133;
      blueValue = 0;
    }
    
      setLed();
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setLed() {
   for (int y = 0; y < pxlCount; y++ ) 
   {
     //setPixel(y, redValue, greenValue, blueValue);
     if ((y>=74) && (y<=118))
     {
       //setPixel(y, 30, 0, 0);
       setPixel(y, int(redValue/5), int(greenValue/5), int(blueValue/5));
     }
     else
     {
       setPixel(y, redValue, greenValue, blueValue);
     }

   }
    strip.show();
    delay(100);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setCheer() {
   for (int y = 0; y < pxlCount; y++ ) {
   setPixel(y, redcheer, greencheer, bluecheer);
   }
    strip.show();
    delay(100);
}


// Fill the dots one after the other with a color
void colorWipe(uint32_t c, uint8_t wait) {
  for(uint16_t i=0; i<strip.numPixels(); i++) {
    strip.setPixelColor(i, c);
    strip.show();
    delay(wait);
  }
}

//////////////////////////////////////////////////
uint32_t ColorPalette(float num) 
{
  switch (palette) {
    case 0: return (num < 0) ? Rainbow(gradient) : Rainbow(num);
    case 1: return (num < 0) ? Sunset(gradient) : Sunset(num);
    case 2: return (num < 0) ? Ocean(gradient) : Ocean(num);
    case 3: return (num < 0) ? PinaColada(gradient) : PinaColada(num);
    case 4: return (num < 0) ? Sulfur(gradient) : Sulfur(num);
    case 5: return (num < 0) ? NoGreen(gradient) : NoGreen(num);
    default: return Rainbow(gradient);
  }
}





void Cycle() 
{
  for (int i = 0; i < strip.numPixels(); i++) {
    float val = float(thresholds[palette]) * (float(i) / float(strip.numPixels())) + (gradient);
    val = int(val) % thresholds[palette];
    strip.setPixelColor(i, ColorPalette(val));
  }
  strip.show();
  gradient += 32;
}




//////////////////////////////////////////////////////////////
void fade(float damper) 
{
  for (int i = 0; i < strip.numPixels(); i++) {
    uint32_t col = strip.getPixelColor(i);
    if (col == 0) continue;
    float colors[3];
    for (int j = 0; j < 3; j++) colors[j] = split(col, j) * damper;
    strip.setPixelColor(i, strip.Color(colors[0] , colors[1], colors[2]));
  }
}


////////////////////////////////////////////////////////////////
void bleed(uint8_t Point) 
{
  for (int i = 1; i < strip.numPixels(); i++) {
    int sides[] = {Point - i, Point + i};

    for (int i = 0; i < 2; i++) {
      int point = sides[i];
      uint32_t colors[] = {strip.getPixelColor(point - 1), strip.getPixelColor(point), strip.getPixelColor(point + 1)  };
      strip.setPixelColor(point, strip.Color(
                             float( split(colors[0], 0) + split(colors[1], 0) + split(colors[2], 0) ) / 3.0,
                             float( split(colors[0], 1) + split(colors[1], 1) + split(colors[2], 1) ) / 3.0,
                             float( split(colors[0], 2) + split(colors[1], 2) + split(colors[2], 2) ) / 3.0)
                          );
    }
  }
}


/////////////////////////////////////////////////////////////////////////////////
uint8_t split(uint32_t color, uint8_t i ) 
{
  if (i == 0) return color >> 16;
  if (i == 1) return color >> 8;
  if (i == 2) return color >> 0;
  return -1;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
uint32_t Rainbow(unsigned int i) 
{
  if (i > 1529) return Rainbow(i % 1530);
  if (i > 1274) return strip.Color(255, 0, 255 - (i % 255));
  if (i > 1019) return strip.Color((i % 255), 0, 255);
  if (i > 764) return strip.Color(0, 255 - (i % 255), 255);
  if (i > 509) return strip.Color(0, 255, (i % 255));
  if (i > 255) return strip.Color(255 - (i % 255), 255, 0);
  return strip.Color(255, i, 0);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
uint32_t Sunset(unsigned int i) {
  if (i > 1019) return Sunset(i % 1020);
  if (i > 764) return strip.Color((i % 255), 0, 255 - (i % 255));
  if (i > 509) return strip.Color(255 - (i % 255), 0, 255);
  if (i > 255) return strip.Color(255, 128 - (i % 255) / 2, (i % 255));
  return strip.Color(255, i / 2, 0);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
uint32_t Ocean(unsigned int i) {
  if (i > 764) return Ocean(i % 765);
  if (i > 509) return strip.Color(0, i % 255, 255 - (i % 255));
  if (i > 255) return strip.Color(0, 255 - (i % 255), 255);
  return strip.Color(0, 255, i);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
uint32_t PinaColada(unsigned int i) {
  if (i > 764) return PinaColada(i % 765);
  if (i > 509) return strip.Color(255 - (i % 255) / 2, (i % 255) / 2, (i % 255) / 2);
  if (i > 255) return strip.Color(255, 255 - (i % 255), 0);
  return strip.Color(128 + (i / 2), 128 + (i / 2), 128 - i / 2);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
uint32_t Sulfur(unsigned int i) {
  if (i > 764) return Sulfur(i % 765);
  if (i > 509) return strip.Color(i % 255, 255, 255 - (i % 255));
  if (i > 255) return strip.Color(0, 255, i % 255);
  return strip.Color(255 - i, 255, 0);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
uint32_t NoGreen(unsigned int i) {
  if (i > 1274) return NoGreen(i % 1275);
  if (i > 1019) return strip.Color(255, 0, 255 - (i % 255));
  if (i > 764) return strip.Color((i % 255), 0, 255);
  if (i > 509) return strip.Color(0, 255 - (i % 255), 255);
  if (i > 255) return strip.Color(255 - (i % 255), 255, i % 255);
  return strip.Color(255, i, 0);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void RGBLoop(int LP) {

  int getOut = 0;
  
  for (int j = 0; j < 3; j++ ) 
  {
    if (getOut) {break;}
    
    // Fade IN
    for (int k = 0; k < 256; k++) {
      if (LP != lastLEDpattern){if (debug) {Serial.println(F("broken out of loop success"));}getOut=1; break;}
      switch (j) {
        case 0: setAll(k, 0, 0); break;
        case 1: setAll(0, k, 0); break;
        case 2: setAll(0, 0, k); break;
      }
      strip.show();
      delay(3);
      if ( j % 50 == 0) {Blynk.run();} //call blynk.run every 50th iteration. Keeps connection alive (hearbeat)
    }
    // Fade OUT
    for (int k = 255; k >= 0; k--) {
      if (LP != lastLEDpattern){if (debug) {Serial.println(F("broken out of loop success"));}getOut=1; break;}
      switch (j) {
        case 0: setAll(k, 0, 0); break;
        case 1: setAll(0, k, 0); break;
        case 2: setAll(0, 0, k); break;
      }
      strip.show();
      delay(3);
      if ( k % 50 == 0) {Blynk.run();}
    }
  }
}

///////////////////////////////////////////////////////////

// Input a value 0 to 255 to get a color value.
// The colours are a transition r - g - b - back to r.
/*
uint32_t Wheel(byte WheelPos) {
  WheelPos = 255 - WheelPos;
  if(WheelPos < 85) {
    return strip.Color(255 - WheelPos * 3, 0, WheelPos * 3);
  }
  if(WheelPos < 170) {
    WheelPos -= 85;
    return strip.Color(0, WheelPos * 3, 255 - WheelPos * 3);
  }
  WheelPos -= 170;
  return strip.Color(WheelPos * 3, 255 - WheelPos * 3, 0);
}
*/
///////////////////////////////////////////////////////////

// Input a value 0 to 255 to get a color value.
// The colors are a transition r - g - b - back to r.
uint32_t Wheel2(byte WheelPos) 
{
  if(WheelPos < 85) {
   return strip.Color(WheelPos * 3, 255 - WheelPos * 3, 0);
  } else if(WheelPos < 170) {
   WheelPos -= 85;
   return strip.Color(255 - WheelPos * 3, 0, WheelPos * 3);
  } else {
   WheelPos -= 170;
   return strip.Color(0, WheelPos * 3, 255 - WheelPos * 3);
  }
}
///////////////////////////////////////////////////////////
byte* Wheel(byte WheelPos) {
  static byte c[3];

  if (WheelPos < 85) {
    c[0] = WheelPos * 3;
    c[1] = 255 - WheelPos * 3;
    c[2] = 0;
  }
  else if (WheelPos < 170) {
    WheelPos -= 85;
    c[0] = 255 - WheelPos * 3;
    c[1] = 0;
    c[2] = WheelPos * 3;
  }
  else {
    WheelPos -= 170;
    c[0] = 0;
    c[1] = WheelPos * 3;
    c[2] = 255 - WheelPos * 3;
  }

  return c;
}
