/*
May 2021
Copyright - www.PlastiBots.com
Rear Parking/Brake LED Strip for vehicle.  Uses 12V (parking light) as main power source, 12V when brakes are applied (as a signal), and GND from the vehicle.
Uses an OptoIsolator to isolate the brake light power from the MCU. The actuation of the brake lights will also illuminate the LED strip at a brighter level to allow it to
enhance the effectiveness and visibility of the rear brake lights

12V main to power a 5V switching regulator then out to the Nano and LED strip. 

Full write-up with circuit at www.plastibots.com

Useful examples/credits:
Bouncing LEDs:  https://github.com/KenKenMkIISR/Bouncing-LED
KITT mode   >>https://learn.adafruit.com/larson-scanner-shades/arduino-code
v7 - STABLE added KITT mode   >>https://learn.adafruit.com/larson-scanner-shades/arduino-code
v6 - removed redundant functions
v5 - cleaned up variables, added functionality
v4 - enhanced fading effect for startup
v3 - added NeoPixel addressable LED strip and sweep out function

*/


#include <Adafruit_NeoPixel.h> //https://github.com/adafruit/Adafruit_NeoPixel

#define LED_NUM 66              // Number of LEDs in pixel strip. DEFAULT 66
#define midPixel round(LED_NUM/2)
#define PixelPin 5                 //Pin for NeoPixel data line
#define BrakePin 3                 //Pin for Brake signal (via Opto)
#define ModePin  2                 //Pin for Mod switch 
#define LEDPin  4                 //Pin for Mod switch 

//FOR BOUNCING LED INTRO
#define BLOCK 8 // LED block length for bouncing effect (number of LEDs) //for car ~8=10 is good. DEFAULT 8
#define GRAVITY 20 // Gravitational acceleration  - default 20
// Bounce coefficient at the lowest point = R1 / R2
#define R1 3 // Bounce coefficient 1  default 3
#define R2 6 // Bounce coefficient 2  default 5  >> 8 is nice
#define REFMINV -80 // Minimum bounce speed
uint8_t col = 1; // Color of falling LED block
//END FOR BOUNCING LED INTRO


#define ledNormVal 180            //normal LED light level.
#define ledMaxVal  250            //Full LED brightness level - to be applied when brakes are on.
byte ledRampSpdMS = 10;           //Configurable ramp speed for LEDS
boolean ledState = 0;             //The current state of the brakes/LEDs.  0=off 1=on
int brakeThresh = 500;            //brake threshold.  Pin is high. When brakes applied, pin drops to near 0V.  value in mV 
int brakeVal = 999;

int pos = 0, dir = 1;             // For KITT:  Position, direction of "eye"

//NeoPixel LEDs
Adafruit_NeoPixel strip = Adafruit_NeoPixel(LED_NUM, PixelPin, NEO_GRB + NEO_KHZ800);


void setup()
{
  //Serial.begin(115200);
  pinMode(PixelPin, OUTPUT);
  pinMode(BrakePin, INPUT);
  pinMode(ModePin, INPUT);
  pinMode(LEDPin, OUTPUT);
  pinMode(9, OUTPUT);
  digitalWrite(9, HIGH);
 
  strip.begin();

  setAll(0, 0, 0);  //turn on the LED sweep at startup.
  delay(2000);
  
  //Do the funky LED intro based on mode (set via toggle switch)
  //on startup of car, we dont want brakes to "ramp up" after the initial sequence if they are already applied, so just go full on
  //if (digitalRead(ModePin) && !digitalRead(BrakePin) == 0)  //not when brakes are applied
  if (digitalRead(ModePin))  
  {
   for (int i = 0; i < LED_NUM*5; i++) //each loop sweeps from one side to the other - do it 8 times
   {
    KITT();
   }
   setAll(0, 0, 0);  //turn on the LED sweep at startup.
   delay(500);
   bounceLEDs(500);
  }
  else
  {
    setAll(ledNormVal, 0, 0);
  }
}  

void loop()
{

  
  //Are brakes being applied? If so, ramp up to high - but only once
  //Optoisolator keeps brakePin Hi at 5V. When 12V applied (brake light), pulls brakePin low.  Hi=1023.  Just picked an arbitrary # of 500
  brakeVal = digitalRead(BrakePin);  //Note val 1 means no brakes (0V) val of 0 means brakes (12V)

  //Serial.print("BrakeRAW="); Serial.print(brakeVal); 
  //Serial.print(" Mode="); Serial.println(digitalRead(ModePin)); 
 
  if(brakeVal == 0 && ledState == 0)  //brakes applied  (digital input is LOW)
  {
    setBrakes(ledMaxVal, 0, 0, 2, 0);
    digitalWrite(LEDPin, HIGH);  
    ledState = 1;
  }
  else if(brakeVal == 1 && ledState == 1)  //brakes not applied - go back to normal (digital input is HIGH)
  {
    setAll(ledNormVal, 0, 0);
    digitalWrite(LEDPin, LOW);
    ledState = 0;
  }
            
  delay(20);  //give the processor some breathing time.       
}  
//END LOOP



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void set_block (uint16_t y, uint8_t c) 
{
  // LED block display
  // y Start position of LED block to be displayed
  // c Display color 0 (off), 1 (red) -8 (green) -16 (blue) -24 (red)
  uint32_t rgb;
  //Serial.print("Val of c="); Serial.println(c); 
  if (c == 0) rgb = strip.Color (0,0,0);
  //else if (c <= 8) rgb = LED.Color (150,0,0); //DJA - to make it all red
  //else if (c <= 8) rgb = strip.Color ((uint16_t) (16-c) * BRIGHT >> 3,0,0); //DJA - to make it all red
  else if (c <= 8) rgb = strip.Color (ledNormVal,0,0); //DJA - to make it all red
  else return;

  for (uint8_t i = 0; i < BLOCK; i++) 
  {
    if (y<0) break;
    //Serial.print("y="); Serial.println(y);
    strip.setPixelColor ((LED_NUM-1)-y, rgb);  //DJA - added for right side sweeping to center
    strip.setPixelColor (y--, rgb);   //default
    //delay(30);
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Intro when car starts up - bounce LEDs from outside to middle.

void bounceLEDs(int returnDelay)
{
  int32_t y; // LED block start position (0 is the top)
  int32_t v; // Fall speed
  int32_t h = (int32_t) LED_NUM << 8; // lowest point position
  h=h/2; //DJA - only want to go to the middle halfway point
  // Bouncing LED loop
  do {
    // LED block initial position, initial speed setting
    y = 0;
    v = 0;
    do {
      set_block ((uint16_t) (y >> 8), col); // LED block display
      strip.show();
      delay(10);
      set_block ((uint16_t) (y >> 8), 0); // Erase LED block
      v = v + GRAVITY; // Fall speed update
      y = y + v; // Update LED block start position
      if (y>= h) {// In case of collision with the lowest point
        y = h-256; // Return to collision position
        v = -v * R1 / R2; // Bounce
        if (v> REFMINV) {// Stop when the bounce speed is less than a certain value
          set_block ((uint16_t) (h >> 8) -1, col); // Fixed display of LED block above the current lowest point
          h = h-(int32_t) (BLOCK * 256); // Raise the lowest point by one block
        }
      }
    } while (y<h); // Repeat while the beginning of the LED block is above the lowest point
  } while (h>=256); // Repeat until the lowest point is less than 1
  strip.show();
  delay(returnDelay);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setPixel(int Pixel, byte red, byte green, byte blue) {
  strip.setPixelColor(Pixel, strip.Color(red, green, blue));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setAll(byte red, byte green, byte blue) 
{
  for (int i = 0; i < LED_NUM; i++ ) 
  {
    //setPixel(i, red, green, blue);
    strip.setPixelColor(i, strip.Color(red, green, blue));
  }
  strip.show(); 
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setBrakesHigh(byte red, byte green, byte blue) 
{
  //int j=midPixel;
  for (int i = 0; i < LED_NUM; i++ ) 
  {
    //setPixel(i, red, green, blue);
    if(!inRange(i, LED_NUM/2-10, LED_NUM/2+10)) //leave the inner strip brightness as is, and brighten the ones closer to the brake lights
    {
      strip.setPixelColor(i, strip.Color(red, green, blue));
    }
  }
  strip.show();   
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setBrakes(byte red, byte green, byte blue, int SpeedDelay, int ReturnDelay) 
{
  int j=midPixel;
  /*  USE THIS TO JUST GO FROM PARKING TO BRAKES ON
  for (int i = 0; i < LED_NUM; i++ ) 
  {
    //setPixel(i, red, green, blue);
    if(!inRange(i, LED_NUM/2-10, LED_NUM/2+10))
    {
      strip.setPixelColor(i, strip.Color(red, green, blue));
    }
  }
  strip.show(); 
  */

  
  for (int i = round(LED_NUM/2)+1; i <= LED_NUM; i++) 
  {
    if(!inRange(i, (LED_NUM/2)-10, (LED_NUM/2)+10))
    {
      setPixel(i+1, red, green, blue);  
      setPixel(i+2, red-20, green, blue);  
      setPixel(i+3, red-40, green, blue);  
      setPixel(i+4, red-60, green, blue);  
      setPixel(i+5, red-80, green, blue);  
  
      setPixel(j-1, red, green, blue);    
      setPixel(j-2, red-20, green, blue);    
      setPixel(j-3, red-40, green, blue);    
      setPixel(j-4, red-60, green, blue);    
      setPixel(j-5, red-80, green, blue);    
    }
    else
    {
      setPixel(i+1, ledNormVal, green, blue); 
      setPixel(i, ledNormVal, green, blue);  
      setPixel(j, ledNormVal, green, blue);    
      setPixel(j-1, ledNormVal, green, blue);    
    }
    j--;
    strip.show();
    delay(SpeedDelay);
  }
  delay(ReturnDelay);
  
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void KITT()
{
  int j;
 
  // Draw 5 pixels centered on pos.  setPixelColor() will clip any
  // pixels off the ends of the strip, we don't need to watch for that.
  strip.setPixelColor(pos - 8, 0x100000); // Dark red
  strip.setPixelColor(pos - 7, 0x100000); // Dark red
  strip.setPixelColor(pos - 6, 0x100000); // Dark red
  strip.setPixelColor(pos - 5, 0x800000); // Medium red
  strip.setPixelColor(pos - 4, 0x800000); // Medium red  
  strip.setPixelColor(pos - 3, 0x800000); // Medium red
  strip.setPixelColor(pos - 2, 0xFF3000); // Center 
  strip.setPixelColor(pos - 1, 0xFF3000); // Center 
  strip.setPixelColor(pos    , 0xFF3000); // Center pixel is brightest
  strip.setPixelColor(pos + 1, 0xFF3000); // Center 
  strip.setPixelColor(pos + 2, 0xFF3000); // Center 
  strip.setPixelColor(pos + 3, 0x800000); // Medium red
  strip.setPixelColor(pos + 4, 0x800000); // Medium red
  strip.setPixelColor(pos + 5, 0x800000); // Medium red
  strip.setPixelColor(pos + 6, 0x100000); // Dark red
  strip.setPixelColor(pos + 7, 0x100000); // Dark red
  strip.setPixelColor(pos + 8, 0x100000); // Dark red
 
  strip.show();
  delay(25);
 
  // Rather than being sneaky and erasing just the tail pixel,
  // it's easier to erase it all and draw a new one next time.
  for(j=-8; j<= 8; j++) strip.setPixelColor(pos+j, 0);  //count is based on the number of pixels set above
 
  // Bounce off ends of strip
  pos += dir;
  if(pos < 0) {
    pos = 1;
    dir = -dir;
  } else if(pos >= strip.numPixels()) {
    pos = strip.numPixels() - 8;
    dir = -dir;
  }
}


bool inRange(int val, int minimum, int maximum)
{
  return ((minimum <= val) && (val <= maximum));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
 
 void CenterToOutside(byte red, byte green, byte blue, int EyeSize, int SpeedDelay, int ReturnDelay) 
{
  for (int i = ((LED_NUM - EyeSize) / 2); i >= 0; i--) 
  {
    setAll(0, 0, 0);
    //if (LP != lastLEDpattern){break;} //allows to break out of looping when Blynk pattern is changed

    setPixel(i, red / 10, green / 10, blue / 10);
    for (int j = 1; j <= EyeSize; j++) {
      setPixel(i + j, red, green, blue);
    }
    setPixel(i + EyeSize + 1, red / 10, green / 10, blue / 10);

    setPixel(LED_NUM - i, red / 10, green / 10, blue / 10);
    for (int j = 1; j <= EyeSize; j++) {
      setPixel(LED_NUM - i - j, red, green, blue);
    }
    setPixel(LED_NUM - i - EyeSize - 1, red / 10, green / 10, blue / 10);
    
    strip.show();
    delay(SpeedDelay);
    //if ( i % 50 == 0) {if (debug) {Serial.println(F("CenterToOutsideWorking"));} Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)    if ( i % 10 == 0) {Blynk.run();} //call blynk.run every 10th iteration. Keeps connection alive (hearbeat)
  }
  delay(ReturnDelay);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void StartUp(byte red, byte green, byte blue, int SpeedDelay, int ReturnDelay) 
{
  int j=midPixel;
  setPixel(midPixel, red, green, blue);
  
  for (int i = round(LED_NUM/2)+1; i <= LED_NUM; i++) 
  {
    //brightness range 220, 180, 120, 80, 50, 20  
    setPixel(i+1, 110, green, blue);  
    setPixel(i+2, 80, green, blue);  
    setPixel(i+3, 50, green, blue);  
    setPixel(i+4, 30, green, blue);  
    setPixel(i+5, 10, green, blue);  
  
    setPixel(j-1, 110, green, blue);    
    setPixel(j-2, 80, green, blue);    
    setPixel(j-3, 50, green, blue);    
    setPixel(j-4, 30, green, blue);    
    setPixel(j-5, 10, green, blue);    
    j--;
    strip.show();


    //strip.show();
    delay(SpeedDelay);
  }
  delay(ReturnDelay);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void BrakesOnOff(byte red, byte green, byte blue, int SpeedDelay, int ReturnDelay) 
{
  //int j=midPixel;
  //setPixel(midPixel, red, 255, blue);
  
  for (int i = 1; i <= LED_NUM; i++) 
  {
    setPixel(i, red, green, blue);  
  }
  strip.show();
  delay(ReturnDelay);
}
*/

//END
