/*
Sept 2021 Copyright - www.PlastiBots.com
Usage: Source code, schematics and build information is provided to maintain the spirit of open source. 
Per that model, you are authorized to re-use, change, modify and update the code as long as original 
credits are maintained and published for public consumption. It may not be used for commercial, re-sale 
or other for-profit purposes unless written consent is provided.

Dynamic Sequential Rear LED Strip for vehicle.  Uses 12V (parking light) as main power source, 12V when brakes are 
applied (as a signal), and GND from the vehicle. Uses an OptoIsolator to isolate the 12 brake power from the MCU. 
The actuation of the brake lights will init a sequential brightening sweep from center to each side.

The system consists of a Master and Slave (this) unit.  The Master is in the center console and controls the
LED sequence options as well as displays status.  It uses a NodeMCU ESP8266 and a OLED screen. The Slave (this) unit
is also a NodeMCU ESP8266.

Design:
12V main to switching regulator drops to 5V.  NodeMCU Mini (ESP8266)and LED strip powered via 5V. 
On boot, the LEDs will follow the last set sequence, then settle into "driving/parking" light mode.  Note, 
the whole system only turns on when the lights come on.

Useful examples/credits:
https://github.com/bnbe-club/esp-now-examples-diy-62  EPS-NOW
https://github.com/KenKenMkIISR/Bouncing-LED
https://github.com/modustrialmaker/Lightsaber_single_blade_neopixel/blob/master/Rei_lightsaber/Rei_lightsaber.ino
https://www.youtube.com/watch?v=4Vh45MSCXBA  Connecting Atom Lite to Blynk BLE
https://docs.m5stack.com/en/core/atom_lite?id=peripherals-pin-map  ATOM Lite Pin Map

TIP!!!  Make sure the wiring between the LED strip and controller is as short as possible. I ran into issues when I swiched from
using an Arduino Nano to an ESP8266.  The original wiring was close to 6 ft and it worked with the Nano. However, after switching
to the ESP8266, it was failing (LEDs were going whacky!).  Shortened the cable to 3ft worked.  

v13 - fixes and adde rainbow fade mode. Paired with SequentialRearLEDsMASTERv6.ino
v12 - Added new sequences
v10,11 - refinements and bug fixes
v9 - Added ESP-NOW functionality to allow master / slave comms. Removed Blynk functionality as it wasnt being used
v8 - Migrated to ESP32 using M5 Stack Atom and Blynk integration
v7 - STABLE added KITT mode   >>https://learn.adafruit.com/larson-scanner-shades/arduino-code
v6 - removed redundant functions
v5 - cleaned up variables, added functionality
v4 - enhanced fading effect for startup
v3 - added NeoPixel addressable LED strip and sweep out function

*/



#include <Adafruit_NeoPixel.h>      // https://github.com/adafruit/Adafruit_NeoPixel
#include <EEPROM.h>                 // for storing variables to memmory to preserve state
#define EEPROM_SIZE 4               // define the number of bytes you want to access
#include <espnow.h>                 // for ESP-NOW
#include <ESP8266WiFi.h>            // for ESP-NOW


boolean debug = false;              //Be sure to set to false when done debugging!!

#define statusLED D5
#define STRIP_LED_NUM 66            // Number of LEDs in pixel strip. DEFAULT 66 for actual strip on MX-5. 24 for testing
#define midPixel round(STRIP_LED_NUM/2)
#define PixelPin D7                 // Pin for NeoPixel data line
#define BrakePin D6                 // Pin for Brake signal (via Opto)

//FOR BOUNCING LED INTRO
#define BLOCK 3                     // LED block length (number of LEDs) //for car ~8=10 is good. DEFAULT 8
#define BRIGHT 80                   //
#define GRAVITY 20                  // Gravitational acceleration  - default 20
                                    // Bounce coefficient at the lowest point = R1 / R2
#define R1 3                        // Bounce coefficient 1  default 3
#define R2 6                        // Bounce coefficient 2  default 5  >> 8 is nice
#define REFMINV -80                 // Minimum bounce speed
uint8_t col = 1;                    // Color of falling LED block
//END FOR BOUNCING LED INTRO

unsigned long lastTime = 0;  
unsigned long timerDelay = 2000;    // send readings timer

int ledNormVal = 80;                //normal LED light level.
int ledMaxVal  = 250;               //Full LED brightness level - to be applied when brakes are on.
byte ledRampSpdMS = 10;             //Configurable ramp speed for LEDS
boolean ledState = 0;               //The current state of the brakes/LEDs.  0=off 1=on
//int flashCounter = 0;
int brakeThresh = 500;              //brake threshold.  Pin is high. When brakes applied, pin drops to near 0V.  value in mV 
int brakeVal = 999;
int pos = 0, dir = 1;               // For KITT:  Position, direction of "eye"
int displayMode = 3;                //Default both Larson and Bounce
int startupDispMode = -1;           //one time sending to Master
int LEDStripModeState = 1;          //LED strip - on, sequencing, braking
boolean doSeq = 0;                  //Triggered from Master to init sequence of LEDs
int modeChange = 0;                 //Triggered from Master.
boolean receivingData = false;

//for ESP-NOW: REPLACE WITH THE MAC Address of the receiving ESP to communicate with
//MASTER  5C:CF:7F:1B:D2:C5
//SLAVE 5C:CF:7F:28:CD:08
uint8_t broadcastAddress[] = {0x5C, 0xCF, 0x7F, 0x1B, 0xD2, 0xC5}; //MASTER - OLED Controller
//uint8_t broadcastAddress[] = {0x5C, 0xCF, 0x7F, 0x28, 0xCD, 0x08}; //SLAVE - LED in rear


//ESP-NOW Structure example to send data  Must match the receiver structure
typedef struct struct_message {
  int a1;
  int b1;
  int c1;
  int d1;
  int e1;
} struct_message;

// ESP-NOW Create a struct_message called myData to hold sensor readings
struct_message myData;


//NeoPixel LEDs
Adafruit_NeoPixel strip = Adafruit_NeoPixel(STRIP_LED_NUM, PixelPin, NEO_GRB + NEO_KHZ800);

// Callback when data is sent
void OnDataSent(uint8_t *mac_addr, uint8_t sendStatus) {
  if (debug){Serial.print("Last Packet Send Status: ");}
  if (sendStatus == 0){
    if (debug){Serial.println("Delivery success");}
  }
  else{
    if (debug){Serial.println("Delivery fail");}
  }
}

// Callback when data is received
void OnDataRecv(uint8_t * mac, uint8_t *incomingData, uint8_t len) {
  receivingData = true;
  memcpy(&myData, incomingData, sizeof(myData));
  if (debug){
    Serial.print("Bytes received: ");
    Serial.println(len);
    Serial.print("a1: ");
    Serial.println(myData.a1);  //not used
    Serial.print("b1: ");
    Serial.println(myData.b1);  //not used
    Serial.print("c1: ");
    Serial.println(myData.c1);  //not used
    Serial.print("d1 (doSeq): ");
    Serial.println(myData.d1);  //used
    Serial.print("e1 (modeChange): ");
    Serial.println(myData.e1);  //not used
    Serial.println();
  }

  doSeq = myData.d1;       //Master triggered LED sequence to start
  modeChange = myData.e1;  //Master triggered a change of mode
  if ((modeChange != displayMode) && (modeChange != 0))  //if the mode has changed, write it to EEPROM
  {
    if (debug){Serial.println("Writing Mode Change to EEPROM");}
    displayMode = modeChange;     //set the new display mode
    EEPROM.write(0, modeChange);  //write val to EEPROM byte # 0.  Note, no need to check if different as the put function only writes if the value is different.
    EEPROM.commit();
  }
  
  //sequence start triggered from Master unit
  if (doSeq==1){doLEDSequence(); doSeq=0;}  

  receivingData = false;
}
 

void setup()
{
  pinMode(PixelPin, OUTPUT);
  pinMode(BrakePin, INPUT);
  pinMode(statusLED, OUTPUT);

  Serial.begin(115200);
  EEPROM.begin(EEPROM_SIZE);       //initialize EEPROM with predefined size
  delay(100);
  displayMode = EEPROM.read(0);    //get the stored mode level.
  startupDispMode = displayMode;
  if (debug){Serial.println();Serial.print("DisplayModeEEPROM=");Serial.println(displayMode);}
   
  WiFi.mode(WIFI_STA);             // Set device as a Wi-Fi Station - ESP-NOW

  // Init ESP-NOW
  if (esp_now_init() != 0) {
     if (debug){Serial.println("Error initializing ESP-NOW");}
    return;
  }

  esp_now_set_self_role(ESP_NOW_ROLE_COMBO);

  // Once ESPNow is successfully Init, we will register for Send CB to
  // get the status of Trasnmitted packet
  esp_now_register_send_cb(OnDataSent);
  
  // Register peer
  esp_now_add_peer(broadcastAddress, ESP_NOW_ROLE_COMBO, 1, NULL, 0); 
   
  // Register for a callback function that will be called when data is received
  esp_now_register_recv_cb(OnDataRecv);
       
  strip.begin();

  setAll(0, 0, 0);  //all LEDs off
  delay(2000);
  sendData();


  //displayMode = 5; //For Testing
  doLEDSequence();  //on startup always do whatever the last sequence was
 
}  

void loop()
{

  //Are brakes being applied? If so, ramp up to high - but only once
  //Optoisolator keeps brakePin Hi at 5V. When 12V applied (brake light), pulls brakePin low.  Hi=1023.  Just picked an arbitrary # of 500
  brakeVal = digitalRead(BrakePin);  //Note val 1 means no brakes (0V) val of 0 means brakes (12V)

  //Serial.print("BrakeRAW="); Serial.print(brakeVal); 
  //Serial.print(" Mode="); Serial.println(digitalRead(ModePin)); 
 
  if(brakeVal == 0 && ledState == 0)  //brakes applied  (digital input is LOW)
  {
    LEDStripModeState = 3;
    sendData();
    setBrakes(ledMaxVal, 0, 0, 6, 0);
    ledState = 1;
  }
  else if(brakeVal == 1 && ledState == 1)  //brakes not applied - go back to normal (digital input is HIGH)
  {
    LEDStripModeState = 1;
    sendData();
    setAll(ledNormVal, 0, 0);
    ledState = 0;
  }
            
  delay(20);  //give the processor some breathing time.     
 
  //send ping status and flash the on-board LED to show it's alive.
  if ((millis() - lastTime) > timerDelay) {
    sendData();
    lastTime = millis();
    digitalWrite(statusLED, HIGH);  
   }
   delay(50);
   digitalWrite(statusLED, LOW);      
}  
//END LOOP

//////////////////////////////////////////////////////////////////////////////////////////////////////////////

void doLEDSequence()
{
  switch (displayMode) 
  {
    case 1:
      LEDStripModeState = 2;
      sendData();   //update Master controller
      doKITT();
      break;
    case 2:
      LEDStripModeState = 2;
      sendData();   //update Master controller
      doBounce();
      break;
    case 3:
      LEDStripModeState = 2;
      sendData();   //update Master controller
      doKITT();
      doBounce();
      break;
    case 4: 
      LEDStripModeState = 2;
      sendData();   //update Master controller
      doColourBounce(500);
      break;

    case 5:   //Rainbow Fade
      LEDStripModeState = 2;
      sendData();   //update Master controller
      rainbowFade(3, 6);
      setAll(ledNormVal, 0, 0);
      break;
    
    case 6:   //DEMO MODE 
      LEDStripModeState = 2;
      sendData();   //update Master controller
      for (int i=0; i<=3; i++) //do them all 2 times
      {
        doKITT();
        delay(500);
        doBounce();
        delay(500);
        doColourBounce(500);
        delay(500);
        rainbowFade(3, 6);
        delay(1000);
        rainbowFade(3, 6);
        delay(1000);
        for (int i=0; i<=3; i++) //pump brakes a few times
        {
          setAll(ledNormVal, 0, 0);
          delay(1000);
          setBrakes(ledMaxVal, 0, 0, 6, 0);  
          delay(2000);
        }
        delay(1000);
      }
      break;

    default:
      LEDStripModeState = 1;
      sendData();   //update Master controller
      setAll(ledNormVal, 0, 0);
      break;
  }
  LEDStripModeState = 1;
  sendData();   //update Master controller
  setAll(ledNormVal, 0, 0);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////

void sendData()
{
    //myData.a1 = 1;  //if it's running, it has power so let Master know.
    //myData.b1 = displayMode;
    myData.c1 = LEDStripModeState;
    myData.d1 = 0;
    myData.e1 = startupDispMode;
    //startupDispMode = -1;
    if (!receivingData)
    {
      esp_now_send(broadcastAddress, (uint8_t *) &myData, sizeof(myData));
    }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void doBounce()
{
  bounceLEDs(500);
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void doKITT()
{
   for (int i = 0; i < STRIP_LED_NUM*5; i++) //each loop sweeps from one side to the other - do it 8 times
   {
    KITT();
   }
   setAll(0, 0, 0);  //turn on the LED sweep at startup.
   delay(500);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void set_block (uint16_t y, uint8_t c) 
{
  // LED block display
  // y Start position of LED block to be displayed
  // c Display color 0 (off), 1 (red) -8 (green) -16 (blue) -24 (red)
  uint32_t rgb;
  //Serial.print("Val of c="); Serial.println(c); 
  if (c == 0) rgb = strip.Color (0,0,0);
  //else if (c <= 8) rgb = LED.Color (150,0,0); //DJA - to make it all red
  //else if (c <= 8) rgb = strip.Color ((uint16_t) (16-c) * BRIGHT >> 3,0,0); //DJA - to make it all red
  else if (c <= 8) rgb = strip.Color (ledNormVal,0,0); //DJA - to make it all red
  else return;

  for (uint8_t i = 0; i < BLOCK; i++) 
  {
    if (y<0) break;
    //Serial.print("y="); Serial.println(y);
    strip.setPixelColor ((STRIP_LED_NUM-1)-y, rgb);  //DJA - added for right side sweeping to center
    strip.setPixelColor (y--, rgb);   //default
    //delay(30);
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Intro when car starts up - bounce LEDs from outside to middle.

void bounceLEDs(int returnDelay)
{
  int32_t y; // LED block start position (0 is the top)
  int32_t v; // Fall speed
  int32_t h = (int32_t) STRIP_LED_NUM << 8; // lowest point position
  h=h/2; //DJA - only want to go to the middle halfway point
  // Bouncing LED loop
  do {
    // LED block initial position, initial speed setting
    y = 0;
    v = 0;
    do {
      set_block ((uint16_t) (y >> 8), col); // LED block display
      strip.show();
      delay(10);
      set_block ((uint16_t) (y >> 8), 0); // Erase LED block
      v = v + GRAVITY; // Fall speed update
      y = y + v; // Update LED block start position
      if (y>= h) {// In case of collision with the lowest point
        y = h-256; // Return to collision position
        v = -v * R1 / R2; // Bounce
        if (v> REFMINV) {// Stop when the bounce speed is less than a certain value
          set_block ((uint16_t) (h >> 8) -1, col); // Fixed display of LED block above the current lowest point
          h = h-(int32_t) (BLOCK * 256); // Raise the lowest point by one block
        }
      }
    } while (y<h); // Repeat while the beginning of the LED block is above the lowest point
  } while (h>=256); // Repeat until the lowest point is less than 1
  strip.show();
  delay(returnDelay);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setPixel(int Pixel, byte red, byte green, byte blue) {
  strip.setPixelColor(Pixel, strip.Color(red, green, blue));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setAll(byte red, byte green, byte blue) 
{
  for (int i = 0; i < STRIP_LED_NUM; i++ ) 
  {
    //setPixel(i, red, green, blue);
    strip.setPixelColor(i, strip.Color(red, green, blue));
  }
  strip.show(); 
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setBrakesHigh(byte red, byte green, byte blue) 
{
  //int j=midPixel;
  for (int i = 0; i < STRIP_LED_NUM; i++ ) 
  {
    //setPixel(i, red, green, blue);
    if(!inRange(i, STRIP_LED_NUM/2-10, STRIP_LED_NUM/2+10)) //leave the inner strip brightness as is, and brighten the ones closer to the brake lights
    {
      strip.setPixelColor(i, strip.Color(red, green, blue));
    }
  }
  strip.show();   
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void setBrakes(byte red, byte green, byte blue, int SpeedDelay, int ReturnDelay) 
{
  int j=midPixel;
  /*  USE THIS TO JUST GO FROM PARKING TO BRAKES ON
  for (int i = 0; i < STRIP_LED_NUM; i++ ) 
  {
    //setPixel(i, red, green, blue);
    if(!inRange(i, STRIP_LED_NUM/2-10, STRIP_LED_NUM/2+10))
    {
      strip.setPixelColor(i, strip.Color(red, green, blue));
    }
  }
  strip.show(); 
  */

 //setBrakes(ledMaxVal, 0, 0, 2, 0);
  
  for (int i = round(STRIP_LED_NUM/2)+1; i <= STRIP_LED_NUM; i++) 
  {
    if(!inRange(i, (STRIP_LED_NUM/2)-10, (STRIP_LED_NUM/2)+10))
    {
      setPixel(i+1, red, green, blue);  
      setPixel(i+2, red-20, green, blue);  
      setPixel(i+3, red-40, green, blue);  
      setPixel(i+4, red-60, green, blue);  
      setPixel(i+5, red-80, green, blue);  
  
      setPixel(j-1, red, green, blue);    
      setPixel(j-2, red-20, green, blue);    
      setPixel(j-3, red-40, green, blue);    
      setPixel(j-4, red-60, green, blue);    
      setPixel(j-5, red-80, green, blue);    
    }
    else
    {
      setPixel(i+1, ledNormVal, green, blue); 
      setPixel(i, ledNormVal, green, blue);  
      setPixel(j, ledNormVal, green, blue);    
      setPixel(j-1, ledNormVal, green, blue);    
    }
    j--;
    strip.show();
    delay(SpeedDelay);
  }
  delay(ReturnDelay);
  
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void KITT()
{
  int j;
 
  // Draw 5 pixels centered on pos.  setPixelColor() will clip any
  // pixels off the ends of the strip, we don't need to watch for that.
  strip.setPixelColor(pos - 8, 0x100000); // Dark red
  strip.setPixelColor(pos - 7, 0x100000); // Dark red
  strip.setPixelColor(pos - 6, 0x100000); // Dark red
  strip.setPixelColor(pos - 5, 0x800000); // Medium red
  strip.setPixelColor(pos - 4, 0x800000); // Medium red  
  strip.setPixelColor(pos - 3, 0x800000); // Medium red
  strip.setPixelColor(pos - 2, 0xFF3000); // Center 
  strip.setPixelColor(pos - 1, 0xFF3000); // Center 
  strip.setPixelColor(pos    , 0xFF3000); // Center pixel is brightest
  strip.setPixelColor(pos + 1, 0xFF3000); // Center 
  strip.setPixelColor(pos + 2, 0xFF3000); // Center 
  strip.setPixelColor(pos + 3, 0x800000); // Medium red
  strip.setPixelColor(pos + 4, 0x800000); // Medium red
  strip.setPixelColor(pos + 5, 0x800000); // Medium red
  strip.setPixelColor(pos + 6, 0x100000); // Dark red
  strip.setPixelColor(pos + 7, 0x100000); // Dark red
  strip.setPixelColor(pos + 8, 0x100000); // Dark red
 
  strip.show();
  delay(25);
 
  // Rather than being sneaky and erasing just the tail pixel,
  // it's easier to erase it all and draw a new one next time.
  for(j=-8; j<= 8; j++) strip.setPixelColor(pos+j, 0);  //count is based on the number of pixels set above
 
  // Bounce off ends of strip
  pos += dir;
  if(pos < 0) {
    pos = 1;
    dir = -dir;
  } else if(pos >= strip.numPixels()) {
    pos = strip.numPixels() - 8;
    dir = -dir;
  }
}


bool inRange(int val, int minimum, int maximum)
{
  return ((minimum <= val) && (val <= maximum));
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void doColourBounce(int returnDelay)
{
  int32_t y; 
  int32_t v; 
  int32_t h=(int32_t)STRIP_LED_NUM<<8; 
  h=h/2; //DJA - only want to go to the middle halfway point
  do{
    y=0;
    v=0;
    do{
      set_ColourBounceBlock((uint16_t)(y>>8),col); 
      strip.show();
      delay(10);
      set_ColourBounceBlock((uint16_t)(y>>8),0); 
      v=v+GRAVITY; 
      y=y+v; 
      if(y>=h){ 
        y=h-256; 
        v=-v*R1/R2; 
        if(v>REFMINV){ 
          set_ColourBounceBlock((uint16_t)(h>>8)-1,col); 
          h=h-(int32_t)(BLOCK*256); 
          col=(col+18)%24+1;
        }
      }
    } while(y<h); 
  } while(h>=256); 
  strip.show();
  delay(returnDelay);

  //After the sequence is all illuminated, drop the LEDs off - this can be removed if you 
  //just want the leds to bounce in.
  y=0; 
  v=0; 
  do{
    y=y+v; 
    v=v+GRAVITY/2; 
    //for(int16_t i=STRIP_LED_NUM-1;i>=(int16_t)(y>>8);i--){
    for(int16_t i=STRIP_LED_NUM/2-1;i>=(int16_t)(y>>8);i--){  //DJA half the pixels so they sweep from both sides
      int32_t y1=((int32_t)i<<8)+(y&0xff)-v; 
      if(y1>=y){
        strip.setPixelColor(i,strip.getPixelColor((uint16_t)(y1>>8)));
        strip.setPixelColor(STRIP_LED_NUM - i,strip.getPixelColor((uint16_t)(y1>>8)));  //DJA the other half
      }
      else{
        strip.setPixelColor(i,strip.Color(0,0,0));
        strip.setPixelColor(STRIP_LED_NUM - i-1,strip.Color(0,0,0)); //DJA the other half
      }
    }
    //strip.setPixelColor(STRIP_LED_NUM/2+2,strip.Color(0,0,0)); //DJA the middle pixel needs to be off
    strip.show();
    delay(10);
  }while(y<(int32_t)STRIP_LED_NUM<<8); 
  delay(returnDelay);
  //END Testing

  
 
}

void set_ColourBounceBlock(uint16_t y,uint8_t c)
{
  //int BlockPixelNum = 8;
  // #define REVERSE 
  uint32_t rgb;
  if(c==0) rgb=strip.Color(0,0,0);
  else if(c<=8)  rgb=strip.Color((uint16_t)(8-c)*BRIGHT>>3 ,(uint16_t)c*BRIGHT>>3, 0);
  else if(c<=16) rgb=strip.Color(0, (uint16_t)(16-c)*BRIGHT>>3 ,(uint16_t)(c-8)*BRIGHT>>3);
  else if(c<=24) rgb=strip.Color((uint16_t)(c-16)*BRIGHT>>3 ,0, (uint16_t)(24-c)*BRIGHT>>3);
  else return;

  for(uint8_t i=0; i<BLOCK; i++){
    if(y<0) break;
    //strip.setPixelColor(STRIP_LED_NUM-1-(y--),rgb);
    //strip.setPixelColor(y--,rgb);
    strip.setPixelColor ((STRIP_LED_NUM-1)-y, rgb);  //DJA - added for right side sweeping to center
    strip.setPixelColor (y--, rgb);   //default

  }
}



//////////////////////////////////////////////////////////////////////////////////////////////////////////////
void rainbowFade(int wait, int rainbowLoops) {
  int fadeVal=0, fadeMax=100;

  // Hue of first pixel runs 'rainbowLoops' complete loops through the color
  // wheel. Color wheel has a range of 65536 but it's OK if we roll over, so
  // just count from 0 to rainbowLoops*65536, using steps of 256 so we
  // advance around the wheel at a decent clip.
  for(uint32_t firstPixelHue = 0; firstPixelHue < rainbowLoops*65536;
    firstPixelHue += 256) {

    for(int i=0; i<round(strip.numPixels()/2); i++) { // For each pixel in strip...

      // Offset pixel hue by an amount to make one full revolution of the
      // color wheel (range of 65536) along the length of the strip
      // (strip.numPixels() steps):
      uint32_t pixelHue = firstPixelHue + (i * 65536L / strip.numPixels());

      // strip.ColorHSV() can take 1 or 3 arguments: a hue (0 to 65535) or
      // optionally add saturation and value (brightness) (each 0 to 255).
      // Here we're using just the three-argument variant, though the
      // second value (saturation) is a constant 255.
      strip.setPixelColor(i, strip.gamma32(strip.ColorHSV(pixelHue, 255,255 * fadeVal / fadeMax)));
      strip.setPixelColor(strip.numPixels()-i, strip.gamma32(strip.ColorHSV(pixelHue, 255,255 * fadeVal / fadeMax)));
    }

    strip.show();
    delay(wait);

    if(firstPixelHue < 65536) {                              // First loop,
      if(fadeVal < fadeMax) fadeVal++;                       // fade in
    } else if(firstPixelHue >= ((rainbowLoops-1) * 65536)) { // Last loop,
      if(fadeVal > 0) fadeVal--;                             // fade out
    } else {
      fadeVal = fadeMax; // Interim loop, make sure fade is at max
    }
  }
  delay(500); // Pause 1/2 second
}

//END
