/*
Reworked by Dave @ PlastiBots.  Full details at www.plastibots.com   
This code/project may not be duplicated for commercial gain.  Additionally, please respect there are many 
contributors to parts that make this possible.  
  Credit to the following: 
  
  James Zahary Sep 12, 2020 Using ESP32-CAM-Video-Recorder-junior library https://github.com/jameszah/ESP32-CAM-Video-Recorder-junior
  jameszah/ESP32-CAM-Video-Recorder-junior is licensed under the GNU General Public License v3.0

  https://randomnerdtutorials.com/esp32-cam-take-photo-display-web-server/
  https://github.com/me-no-dev/ESPAsyncWebServer

  What does it do?  Baseline code uses feature to write an AVI file frame-by-frame.  It has been modified from James'
  original version to snap a frame each time the LDR sensor is triggered by the UV light coming on.  This coincides
  with each layer being cured by the Resin printer (Mars 3 Ultra in my case).  The process is controlled via a Blynk
  app that has a few simple features.  It has a terminal window which shows status updates from the ESP32 CAM board.
  It has a slider to adjust for a delayed start from 0-60 minutes.  Useful given there's really not much to record
   for the first hundred or so layers.  Clicking start will start watching the LDR sensor and a frame will be recorded
   each time the UV light comes on.  The terminal will print a notice for the first 10 layers, then every 10th 
   layer.  The front LED also blinks dimly (a flash of sorts) with each frame capture.  The AVI file can be completed
   via a number of ways.  First, when the time runs out, or if you click the Stop button (Blynk), or when the print is done
   the UV light is off for > a defined period of time (120 seconds right now). It watches for this and stops the AVI, and closes the 
   file.

  The is Arduino code, with standard setup for ESP32-CAM
    - Uses AI Thinker ESP32 CAM in Boards Mgr
    - Can also use ESP32 Wrover Module: Partition Scheme Huge APP (3MB No OTA) but it may not work with this version and has not been tested

  Possible issues:
    - Adding the ESPAsyncWebServer and ability to take a photo for previewing may be causing some issues.
      If so, it's easily removed.  Most of it is in webphoto.h and just pull out related code in this file


#######################################################################################################    
V E R S I O N  H I S T O R Y
v13: Clean up issue with crashing and Blynk disconnects.  
v12: JPG capture via web page to view cam ahead of starting print. https://randomnerdtutorials.com/esp32-cam-take-photo-display-web-server/
v11: Added delay start feature - controlled by Blynk
v10: Cleaned up some stuff
v9:  Cleaned up debugging (able to turn off/on)
v8:  Adding functionality for Blynk control. e.g. Recording:  start/stop/new video.
v7:  Added Blynk / wifi connectivity and solved the file naming with date (actual file date is still 1980)
v5/6: Refinements in timing of picture, flash etc
v4: Cleaned up includes, variable declarations etc
v4: Cleaned up camera loop and other bits - didn't like holding in a while statement
v3: Trying GPIO 33 for the LDR sensor. Worked in v1 for #4 (white LED). 33 is the red small on-baord LED
#######################################################################################################    


*/

//#######################################################################################################    
//D E B U G G I N G

#define Lots_of_Stats false  //prints debug info to a log file on the SD card
#define debug false          //used for Serial Print debug statements.

//#######################################################################################################    

#include <movingAvg.h>   
//#define LOG_LOCAL_LEVEL ESP_LOG_VERBOSE
#include "esp_log.h"
//#include "esp_http_server.h"
#include "esp_camera.h"
#include "sensor.h"
#include <stdio.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "esp_system.h"
#include "nvs_flash.h"
#include "nvs.h"
#include "soc/soc.h"             // Disable brownour problems
#include "soc/cpu.h"           
#include "soc/rtc_cntl_reg.h"  // Disable brownour problems

// MicroSD
#include "driver/sdmmc_host.h"
#include "driver/sdmmc_defs.h"
#include "sdmmc_cmd.h"
#include "esp_vfs_fat.h"
#include "FS.h"
#include <SD_MMC.h>

//for photo server capture
#include <ESPAsyncWebServer.h>
AsyncWebServer photoServer(80);  // Create AsyncWebServer object on port 80
#define FILE_PHOTO "/photo.jpg"  // Photo File Name to save on SD
boolean takeNewPhoto = false;
#include "webphoto.h"             //keeping core code separate to keep it clean
//END for photo server capture


#include <WiFi.h>
#include <WiFiClient.h>
#include <BlynkSimpleEsp32.h>
#include <TimeLib.h>
#include <SimpleTimer.h>      //https://playground.arduino.cc/Code/SimpleTimer/
#include <WidgetRTC.h>

//#######################################################################################################    
//W I F I  A N D  B L Y N K  C R E D E N T I A L S
//WiFi credentials.
char auth[] = "S0bwXHolFqLfqOPqC_LxZT1yO6QPNtQY";
char ssid[] = "Thinair";
char pass[] = "GiMMieWiFi5185";
//#######################################################################################################    


#define camFlashPIN 4     // Front LED
#define LDRSensorPIN 33   // Shared with onboard red LED. 
int avgLDR = 0;           // Average value of the LDR sensor reading
int pc = 0;
//char input;   //DJA
boolean frameCapture = false;
static const char vernum[] = "v10";
char devname[30];
String devstr =  "Mars3CAM";

int IncludeInternet = 0;      // NOT USED    0 for no internet, 1 for time only, 2 streaming with WiFiMan, 3 ssid in file, 4 default internet on and file


// https://sites.google.com/a/usapiens.com/opnode/time-zones  -- find your timezone here
String TIMEZONE = "EST5EDT,M3.2.0,M11.1.0";

//Superseeded by the values in config.txt file if exist in root of SD card. If not there, it will use these:
int framesize = FRAMESIZE_HD;     // framesize 8=vga, 9=svga, 10=xga, 11=hd, 12=sxga, 13=uxga, 14=fhd, 17=qxga, 18=qhd, 21=qsxga 
int quality = 2;                  // DJA 6 is ok and default    quality 0-63, lower the better, 10 good start, must be higher than "quality config"
int framesizeconfig = FRAMESIZE_UXGA;  // frame config - must be equal or higher than framesize
int qualityconfig = 1;            // DJA 5 is ok and default    quality config - high q 0..5, med q 6..10, low q 11+
int buffersconfig = 3;            // buffers - 1 is half speed of 3
int avi_length = 14400;            // how long a movie in seconds -- 1800 sec = 30 min 7200 = 2hrs
int frame_interval = 0;           // 0 record at full speed
int speed_up_factor = 24;         // speedup - multiply framerate - 1 for realtime, 24 for record at 1fps, play at 24fps or24x
int stream_delay = 500;           // streamdelay - ms between streaming frames - 0 for fast as possible, 500 for 2fps 
int MagicNumber = 12;             // change this number to reset the eprom in your esp32 for file numbers

bool configfile = false;
bool InternetOff = true;
bool reboot_now = false;
String cssid;
String cpass;
String czone;

static esp_err_t cam_err;
float most_recent_fps = 0;
int most_recent_avg_framesize = 0;

uint8_t* framebuffer;
int framebuffer_len;

int first = 1;
long frame_start = 0;
long frame_end = 0;
long frame_total = 0;
long frame_average = 0;
long loop_average = 0;
long loop_total = 0;
long total_frame_data = 0;
long last_frame_length = 0;
int done = 0;
long avi_start_time = 0;
long avi_end_time = 0;
int start_record = 0;         //used to set recording to start. Controlled by Blynk

int we_are_already_stopped = 0;
long total_delay = 0;
long bytes_before_last_100_frames = 0;
long time_before_last_100_frames = 0;

long time_in_loop = 0;
long time_in_camera = 0;
long time_in_sd = 0;
long time_in_good = 0;
long time_total = 0;
long time_in_web1 = 0;
long time_in_web2 = 0;
long delay_wait_for_sd = 0;
long wait_for_cam = 0;

int do_it_now = 0;
int dsTimer;                               //timerID
long record_delay_start = 1000;  //Default to immediately (1 second) 1000 ms
boolean delay_start = false;
SimpleTimer delayStartTimer;                         //timer for delayed start


long current_frame_time;
long last_frame_time;                                                                     
long time_to_wait_till_print_done = 120 * 1000;  //120 seconds. Need to allot for time for Z to move + time for exposure. Also consider when print is large and z has to move all the way up. Assumption is print is done if no transition from UV on to off for X seconds

// https://github.com/espressif/esp32-camera/issues/182
#define fbs 8 // was 64 -- how many kb of static ram for psram -> sram buffer for sd write
uint8_t framebuffer_static[fbs * 1024 + 20];

// CAMERA_MODEL_AI_THINKER
#define PWDN_GPIO_NUM     32
#define RESET_GPIO_NUM    -1
#define XCLK_GPIO_NUM      0
#define SIOD_GPIO_NUM     26
#define SIOC_GPIO_NUM     27
#define Y9_GPIO_NUM       35                   
#define Y8_GPIO_NUM       34
#define Y7_GPIO_NUM       39
#define Y6_GPIO_NUM       36
#define Y5_GPIO_NUM       21
#define Y4_GPIO_NUM       19
#define Y3_GPIO_NUM       18                                      
#define Y2_GPIO_NUM        5
#define VSYNC_GPIO_NUM    25
#define HREF_GPIO_NUM     23
#define PCLK_GPIO_NUM     22

camera_fb_t * fb_curr = NULL;
camera_fb_t * fb_next = NULL;

char avi_file_name[100];

static int i = 0;
uint16_t frame_cnt = 0;
uint16_t remnant = 0;
uint32_t length = 0;
uint32_t startms;
uint32_t elapsedms;
uint32_t uVideoLen = 0;

int bad_jpg = 0;
int extend_jpg = 0;
int normal_jpg = 0;

int file_number = 0;
int file_group = 0;
long boot_time = 0;

long totalp;
long totalw;

#define BUFFSIZE 512

uint8_t buf[BUFFSIZE];

#define AVIOFFSET 240 // AVI main header length

unsigned long movi_size = 0;
unsigned long jpeg_size = 0;
unsigned long idx_offset = 0;

uint8_t zero_buf[4] = {0x00, 0x00, 0x00, 0x00};
uint8_t dc_buf[4] = {0x30, 0x30, 0x64, 0x63};    // "00dc"
uint8_t dc_and_zero_buf[8] = {0x30, 0x30, 0x64, 0x63, 0x00, 0x00, 0x00, 0x00};

uint8_t avi1_buf[4] = {0x41, 0x56, 0x49, 0x31};    // "AVI1"
uint8_t idx1_buf[4] = {0x69, 0x64, 0x78, 0x31};    // "idx1"

struct frameSizeStruct {
  uint8_t frameWidth[2];
  uint8_t frameHeight[2];
};

//  data structure from here https://github.com/s60sc/ESP32-CAM_MJPEG2SD/blob/master/avi.cpp, extended for ov5640
static const frameSizeStruct frameSizeData[] = {
  {{0x60, 0x00}, {0x60, 0x00}}, // FRAMESIZE_96X96,    // 96x96
  {{0xA0, 0x00}, {0x78, 0x00}}, // FRAMESIZE_QQVGA,    // 160x120
  {{0xB0, 0x00}, {0x90, 0x00}}, // FRAMESIZE_QCIF,     // 176x144
  {{0xF0, 0x00}, {0xB0, 0x00}}, // FRAMESIZE_HQVGA,    // 240x176
  {{0xF0, 0x00}, {0xF0, 0x00}}, // FRAMESIZE_240X240,  // 240x240
  {{0x40, 0x01}, {0xF0, 0x00}}, // FRAMESIZE_QVGA,     // 320x240   framessize
  {{0x90, 0x01}, {0x28, 0x01}}, // FRAMESIZE_CIF,      // 400x296       bytes per buffer required in psram - quality must be higher number (lower quality) than config quality
  {{0xE0, 0x01}, {0x40, 0x01}}, // FRAMESIZE_HVGA,     // 480x320       low qual  med qual  high quality
  {{0x80, 0x02}, {0xE0, 0x01}}, // FRAMESIZE_VGA,      // 640x480   8   11+   ##  6-10  ##  0-5         indoor(56,COUNT=3)  (56,COUNT=2)          (56,count=1)
                                                       //               38,400    61,440    153,600 
  {{0x20, 0x03}, {0x58, 0x02}}, // FRAMESIZE_SVGA,     // 800x600   9
  {{0x00, 0x04}, {0x00, 0x03}}, // FRAMESIZE_XGA,      // 1024x768  10
  {{0x00, 0x05}, {0xD0, 0x02}}, // FRAMESIZE_HD,       // 1280x720  11  115,200   184,320   460,800     (11)50.000  25.4fps   (11)50.000 12fps    (11)50,000  12.7fps
  {{0x00, 0x05}, {0x00, 0x04}}, // FRAMESIZE_SXGA,     // 1280x1024 12
  {{0x40, 0x06}, {0xB0, 0x04}}, // FRAMESIZE_UXGA,     // 1600x1200 13  240,000   384,000   960,000
  // 3MP Sensors
  {{0x80, 0x07}, {0x38, 0x04}}, // FRAMESIZE_FHD,      // 1920x1080 14  259,200   414,720   1,036,800   (11)210,000 5.91fps
  {{0xD0, 0x02}, {0x00, 0x05}}, // FRAMESIZE_P_HD,     //  720x1280 15
  {{0x60, 0x03}, {0x00, 0x06}}, // FRAMESIZE_P_3MP,    //  864x1536 16
  {{0x00, 0x08}, {0x00, 0x06}}, // FRAMESIZE_QXGA,     // 2048x1536 17  393,216   629,146   1,572,864
  // 5MP Sensors
  {{0x00, 0x0A}, {0xA0, 0x05}}, // FRAMESIZE_QHD,      // 2560x1440 18  460,800   737,280   1,843,200   (11)400,000 3.5fps    (11)330,000 1.95fps
  {{0x00, 0x0A}, {0x40, 0x06}}, // FRAMESIZE_WQXGA,    // 2560x1600 19
  {{0x38, 0x04}, {0x80, 0x07}}, // FRAMESIZE_P_FHD,    // 1080x1920 20
  {{0x00, 0x0A}, {0x80, 0x07}}  // FRAMESIZE_QSXGA,    // 2560x1920 21  614,400   983,040   2,457,600   (15)425,000 3.25fps   (15)382,000 1.7fps  (15)385,000 1.7fps

};

const int avi_header[AVIOFFSET] PROGMEM = {
  0x52, 0x49, 0x46, 0x46, 0xD8, 0x01, 0x0E, 0x00, 0x41, 0x56, 0x49, 0x20, 0x4C, 0x49, 0x53, 0x54,
  0xD0, 0x00, 0x00, 0x00, 0x68, 0x64, 0x72, 0x6C, 0x61, 0x76, 0x69, 0x68, 0x38, 0x00, 0x00, 0x00,
  0xA0, 0x86, 0x01, 0x00, 0x80, 0x66, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
  0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x80, 0x02, 0x00, 0x00, 0xe0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4C, 0x49, 0x53, 0x54, 0x84, 0x00, 0x00, 0x00,
  0x73, 0x74, 0x72, 0x6C, 0x73, 0x74, 0x72, 0x68, 0x30, 0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x73,
  0x4D, 0x4A, 0x50, 0x47, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x73, 0x74, 0x72, 0x66,
  0x28, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0xe0, 0x01, 0x00, 0x00,
  0x01, 0x00, 0x18, 0x00, 0x4D, 0x4A, 0x50, 0x47, 0x00, 0x84, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x4E, 0x46, 0x4F,
  0x10, 0x00, 0x00, 0x00, 0x6A, 0x61, 0x6D, 0x65, 0x73, 0x7A, 0x61, 0x68, 0x61, 0x72, 0x79, 0x20,
  0x76, 0x35, 0x30, 0x20, 0x4C, 0x49, 0x53, 0x54, 0x00, 0x01, 0x0E, 0x00, 0x6D, 0x6F, 0x76, 0x69,
};



TaskHandle_t the_camera_loop_task;
TaskHandle_t the_sd_loop_task;
//TaskHandle_t the_streaming_loop_task;

SemaphoreHandle_t wait_for_sd;
SemaphoreHandle_t sd_go;

//  Avi Writer Stuff here
File logfile;
File avifile;
File idxfile;


movingAvg LDRSenReading(4); 
//Blynk RTC widget
WidgetRTC rtc;
WidgetTerminal terminal(V0);   //for logging to terminal window

//
// Writes an uint32_t in Big Endian at current file position
//
static void inline print_quartet(unsigned long i, File fd) {

  uint8_t y[4];
  y[0] = i % 0x100;
  y[1] = (i >> 8) % 0x100;
  y[2] = (i >> 16) % 0x100;
  y[3] = (i >> 24) % 0x100;
  size_t i1_err = fd.write(y , 4);
}

//
// Writes 2 uint32_t in Big Endian at current file position
//
static void inline print_2quartet(unsigned long i, unsigned long j, File fd) {

  uint8_t y[8];
  y[0] = i % 0x100;
  y[1] = (i >> 8) % 0x100;
  y[2] = (i >> 16) % 0x100;
  y[3] = (i >> 24) % 0x100;
  y[4] = j % 0x100;
  y[5] = (j >> 8) % 0x100;
  y[6] = (j >> 16) % 0x100;
  y[7] = (j >> 24) % 0x100;
  size_t i1_err = fd.write(y , 8);
}

//
// if we have no camera, or sd card, then flash rear led on and off to warn the human SOS - SOS
//
void major_fail() {

   if (debug) {Serial.println("");}
  logfile.close();

  for  (int i = 0;  i < 10; i++) {                 // 10 loops or about 100 seconds then reboot
    //for (int j = 0; j < 3; j++) {
    //  digitalWrite(33, LOW);   delay(150);
    //  digitalWrite(33, HIGH);  delay(150);
    //}
    //delay(1000);

    //for (int j = 0; j < 3; j++) {
    //  digitalWrite(33, LOW);  delay(500);
    //  digitalWrite(33, HIGH); delay(500);
    //}
    delay(1000);
     if (debug) {Serial.print("Major Fail  "); Serial.print(i); Serial.print(" / "); Serial.println(10);}
  }

  ESP.restart();
}

//#######################################################################################################    

static esp_err_t config_camera() {

  camera_config_t config;

  //Serial.println("config camera");

  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = Y2_GPIO_NUM;
  config.pin_d1 = Y3_GPIO_NUM;
  config.pin_d2 = Y4_GPIO_NUM;
  config.pin_d3 = Y5_GPIO_NUM;
  config.pin_d4 = Y6_GPIO_NUM;
  config.pin_d5 = Y7_GPIO_NUM;
  config.pin_d6 = Y8_GPIO_NUM;
  config.pin_d7 = Y9_GPIO_NUM;
  config.pin_xclk = XCLK_GPIO_NUM;
  config.pin_pclk = PCLK_GPIO_NUM;
  config.pin_vsync = VSYNC_GPIO_NUM;
  config.pin_href = HREF_GPIO_NUM;
  config.pin_sscb_sda = SIOD_GPIO_NUM;
  config.pin_sscb_scl = SIOC_GPIO_NUM;
  config.pin_pwdn = PWDN_GPIO_NUM;
  config.pin_reset = RESET_GPIO_NUM;

  config.xclk_freq_hz = 20000000;     // 10000000 or 20000000 -- 100 is faster with v1.04  // 200 is faster with v1.06 // 16500000 is an option

  config.pixel_format = PIXFORMAT_JPEG;

   if (debug) {Serial.printf("Frame config %d, quality config %d, buffers config %d\n", framesizeconfig, qualityconfig, buffersconfig);}
  config.frame_size =  (framesize_t)framesizeconfig;
  config.jpeg_quality = qualityconfig;
  config.fb_count = buffersconfig;


  if (debug) {
    Serial.printf("Before camera config ...");
    Serial.printf("Internal Total heap %d, internal Free Heap %d, ", ESP.getHeapSize(), ESP.getFreeHeap());
    Serial.printf("SPIRam Total heap   %d, SPIRam Free Heap   %d\n", ESP.getPsramSize(), ESP.getFreePsram());
  }
  esp_err_t cam_err = ESP_FAIL;
  int attempt = 5;
  while (attempt && cam_err != ESP_OK) {
    cam_err = esp_camera_init(&config);
    if (cam_err != ESP_OK) {
       if (debug) {Serial.printf("Camera init failed with error 0x%x", cam_err);}
      digitalWrite(PWDN_GPIO_NUM, 1);
      delay(500);
      digitalWrite(PWDN_GPIO_NUM, 0); // power cycle the camera (OV2640)
      attempt--;
    }
  }

  if (debug) {
    Serial.printf("After  camera config ...");
    Serial.printf("Internal Total heap %d, internal Free Heap %d, ", ESP.getHeapSize(), ESP.getFreeHeap());
    Serial.printf("SPIRam Total heap   %d, SPIRam Free Heap   %d\n", ESP.getPsramSize(), ESP.getFreePsram());
  }

  if (cam_err != ESP_OK) {
    major_fail();
  }

  sensor_t * ss = esp_camera_sensor_get();

  ///ss->set_hmirror(ss, 1);        // 0 = disable , 1 = enable
  ///ss->set_vflip(ss, 1);          // 0 = disable , 1 = enable


//Reference for camera settings change:  https://randomnerdtutorials.com/esp32-cam-ov2640-camera-settings/
/*
ss->set_brightness(ss, 0);     // -2 to 2
ss->set_contrast(ss, 0);       // -2 to 2
ss->set_saturation(ss, 0);     // -2 to 2
ss->set_special_effect(ss, 0); // 0 to 6 (0 - No Effect, 1 - Negative, 2 - Grayscale, 3 - Red Tint, 4 - Green Tint, 5 - Blue Tint, 6 - Sepia)
ss->set_whitebal(ss, 1);       // 0 = disable , 1 = enable
ss->set_awb_gain(ss, 1);       // 0 = disable , 1 = enable
ss->set_wb_mode(ss, 0);        // 0 to 4 - if awb_gain enabled (0 - Auto, 1 - Sunny, 2 - Cloudy, 3 - Office, 4 - Home)
ss->set_exposure_ctrl(ss, 1);  // 0 = disable , 1 = enable
ss->set_aec2(ss, 0);           // 0 = disable , 1 = enable
ss->set_ae_level(ss, 0);       // -2 to 2
ss->set_aec_value(ss, 300);    // 0 to 1200
ss->set_gain_ctrl(ss, 1);      // 0 = disable , 1 = enable
ss->set_agc_gain(ss, 0);       // 0 to 30
ss->set_gainceiling(ss, (gainceiling_t)0);  // 0 to 6
ss->set_bpc(ss, 0);            // 0 = disable , 1 = enable
ss->set_wpc(ss, 1);            // 0 = disable , 1 = enable
ss->set_raw_gma(ss, 1);        // 0 = disable , 1 = enable
ss->set_lenc(ss, 1);           // 0 = disable , 1 = enable
ss->set_hmirror(ss, 0);        // 0 = disable , 1 = enable
ss->set_vflip(ss, 0);          // 0 = disable , 1 = enable
ss->set_dcw(ss, 1);            // 0 = disable , 1 = enable
ss->set_colorbar(ss, 0);       // 0 = disable , 1 = enable
*/
   if (debug) {Serial.printf("\nCamera started correctly, Type is %x (hex) of 9650, 7725, 2640, 3660, 5640\n\n", ss->id.PID);}

  if (ss->id.PID == OV5640_PID ) {
    //Serial.println("56 - going mirror");
    ss->set_hmirror(ss, 1);        // 0 = disable , 1 = enable
  } else {
    ss->set_hmirror(ss, 0);        // 0 = disable , 1 = enable
  }

  ss->set_quality(ss, quality);
  ss->set_framesize(ss, (framesize_t)framesize);

  ss->set_brightness(ss, 1);  //up the blightness just a bit
  ss->set_saturation(ss, -2); //lower the saturation

  delay(800);
  for (int j = 0; j < 4; j++) {
    camera_fb_t * fb = esp_camera_fb_get(); // get_good_jpeg();
    if (!fb) {
       if (debug) {Serial.println("Camera Capture Failed");}
    } else {
       if (debug) {Serial.print("Pic, len="); Serial.print(fb->len);}
       if (debug) {Serial.printf(", new fb %X\n", (long)fb->buf);}
      esp_camera_fb_return(fb);
      delay(50);
    }
  }
}

//#######################################################################################################    


static esp_err_t init_sdcard()
{

  int succ = SD_MMC.begin("/sdcard", true);
  if (succ) {
     if (debug) {Serial.printf("SD_MMC Begin: %d\n", succ);}
    uint8_t cardType = SD_MMC.cardType();
     if (debug) {Serial.print("SD_MMC Card Type: ");}
    if (cardType == CARD_MMC) {
       if (debug) {Serial.println("MMC");}
    } else if (cardType == CARD_SD) {
       if (debug) {Serial.println("SDSC");}
    } else if (cardType == CARD_SDHC) {
       if (debug) {Serial.println("SDHC");}
    } else {
       if (debug) {Serial.println("UNKNOWN");}
    }

    uint64_t cardSize = SD_MMC.cardSize() / (1024 * 1024);
     if (debug) {Serial.printf("SD_MMC Card Size: %lluMB\n", cardSize);}

  } else {
     if (debug) {Serial.printf("Failed to mount SD card VFAT filesystem. \n");}
     if (debug) {Serial.println("Do you have an SD Card installed?");}
     if (debug) {Serial.println("Check pin 12 and 13, not grounded, or grounded with 10k resistors!\n\n");}
    major_fail();
  }

  return ESP_OK;
}

//#######################################################################################################    
void read_config_file() {
  
// put a file "config.txt" onto SD card, to set parameters different from your hardcoded parameters
// it should look like this - one paramter per line, in the correct order, followed by 2 spaces, and any comments you choose
/*
desklens  // camera name for files, mdns, etc
11  // framesize 9=svga, 10=xga, 11=hd, 12=sxga, 13=uxga, 14=fhd, 17=qxga, 18=qhd, 21=qsxga 
8  // quality 0-63, lower the better, 10 good start, must be higher than "quality config"
11  // framesize config - must be equal or higher than framesize
5  / quality config - high q 0..5, med q 6..10, low q 11+
3  // buffers - 1 is half speed of 3, but you might run out od memory with 3 and framesize > uxga
900  // length of video in seconds
0  // interval - ms between frames - 0 for fastest, or 500 for 2fps, 10000 for 10 sec/frame
1  // speedup - multiply framerate - 1 for realtime, 24 for record at 1fps, play at 24fps or24x
0  // streamdelay - ms between streaming frames - 0 for fast as possible, 500 for 2fps 
4  // 0 no internet, 1 get time then shutoff, 2 streaming using wifiman, 3 for use ssid names below default off, 4 names below default on
MST7MDT,M3.2.0/2:00:00,M11.1.0/2:00:00  // timezone - this is mountain time, find timezone here https://sites.google.com/a/usapiens.com/opnode/time-zones
ssid1234  // ssid
mrpeanut  // ssid password

Lines above are rigid - do not delete lines, must have 2 spaces after the number or string
*/

  File config_file = SD_MMC.open("/config.txt", "r");
  if (!config_file) {
     if (debug) {Serial.println("Failed to open config_file for reading");}
  } else {
    String junk;
     if (debug) {Serial.println("Reading config.txt");}
    String cname = config_file.readStringUntil(' ');
    junk = config_file.readStringUntil('\n');
    int cframesize = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cquality = config_file.parseInt();
    junk = config_file.readStringUntil('\n');

    int cframesizeconfig = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cqualityconfig = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cbuffersconfig = config_file.parseInt();
    junk = config_file.readStringUntil('\n');

    int clength = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cinterval = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cspeedup = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cstreamdelay = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    int cinternet = config_file.parseInt();
    junk = config_file.readStringUntil('\n');
    String czone = config_file.readStringUntil(' ');
    junk = config_file.readStringUntil('\n');
    cssid = config_file.readStringUntil(' ');
    junk = config_file.readStringUntil('\n');
    cpass = config_file.readStringUntil(' ');
    junk = config_file.readStringUntil('\n');
    config_file.close();

    if (debug)
    {
      Serial.printf("=========   Data fram config.txt   =========\n");
      Serial.printf("Name %s\n", cname); logfile.printf("Name %s\n", cname);
      Serial.printf("Framesize %d\n", cframesize); logfile.printf("Framesize %d\n", cframesize);
      Serial.printf("Quality %d\n", cquality); logfile.printf("Quality %d\n", cquality);
      Serial.printf("Framesize config %d\n", cframesizeconfig); logfile.printf("Framesize config%d\n", cframesizeconfig);
      Serial.printf("Quality config %d\n", cqualityconfig); logfile.printf("Quality config%d\n", cqualityconfig);
      Serial.printf("Buffers config %d\n", cbuffersconfig); logfile.printf("Buffers config %d\n", cbuffersconfig);
      Serial.printf("Length %d\n", clength); logfile.printf("Length %d\n", clength);
      Serial.printf("Interval %d\n", cinterval); logfile.printf("Interval %d\n", cinterval);
      Serial.printf("Speedup %d\n", cspeedup); logfile.printf("Speedup %d\n", cspeedup);
      Serial.printf("Streamdelay %d\n", cstreamdelay); logfile.printf("Streamdelay %d\n", cstreamdelay);
      Serial.printf("Internet %d\n", cinternet); logfile.printf("Internet %d\n", cinternet);
      //Serial.printf("Zone len %d, %s\n", czone.length(), czone); //logfile.printf("Zone len %d, %s\n", czone.length(), czone);
      Serial.printf("Zone len %d\n", czone.length()); logfile.printf("Zone len %d\n", czone.length());
      Serial.printf("ssid %s\n", cssid); logfile.printf("ssid %s\n", cssid);
      Serial.printf("pass %s\n", cpass); logfile.printf("pass %s\n", cpass);
    }


    framesize = cframesize;
    quality = cquality;
    framesizeconfig = cframesizeconfig;
    qualityconfig = cqualityconfig;
    buffersconfig = cbuffersconfig;
    avi_length = clength;
    frame_interval = cinterval;
    speed_up_factor = cspeedup;
    stream_delay = cstreamdelay;
    IncludeInternet = cinternet;
    configfile = true;
    TIMEZONE = czone;

    cname.toCharArray(devname, cname.length() + 1);
  }
}

//#######################################################################################################    
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//  delete_old_stuff() - delete oldest files to free diskspace
//

void listDir( const char * dirname, uint8_t levels) {

   if (debug) {Serial.printf("Listing directory: %s\n", "/");}

  File root = SD_MMC.open("/");
  if (!root) {
     if (debug) {Serial.println("Failed to open directory");}
    return;
  }
  if (!root.isDirectory()) {
     if (debug) {Serial.println("Not a directory");}
    return;
  }

  File filex = root.openNextFile();
  while (filex) {
    if (filex.isDirectory()) {
       if (debug) {Serial.print("  DIR : ");}
       if (debug) {Serial.println(filex.name());}
      if (levels) {
        listDir( filex.name(), levels - 1);
      }
    } else {
       if (debug) {Serial.print("  FILE: ");}
       if (debug) {Serial.print(filex.name());}
       if (debug) {Serial.print("  SIZE: ");}
       if (debug) {Serial.println(filex.size());}
    }
    filex = root.openNextFile();
  }
}

//#######################################################################################################    
void delete_old_stuff() {

   if (debug) {Serial.printf("Total space: %lluMB\n", SD_MMC.totalBytes() / (1024 * 1024));}
   if (debug) {Serial.printf("Used space: %lluMB\n", SD_MMC.usedBytes() / (1024 * 1024));}

  //listDir( "/", 0);

  float full = 1.0 * SD_MMC.usedBytes() / SD_MMC.totalBytes();
  if (full  <  0.8) {
     if (debug) {Serial.printf("Nothing deleted, %.1f%% disk full\n", 100.0 * full);}
  } else {
     if (debug) {Serial.printf("Disk is %.1f%% full ... deleting oldest file\n", 100.0 * full);}
    while (full > 0.8) {

      double del_number = 999999999;
      char del_numbername[50];

      File f = SD_MMC.open("/");

      File file = f.openNextFile();

      while (file) {
        //Serial.println(file.name());
        if (!file.isDirectory()) {

          char foldname[50];
          strcpy(foldname, file.name());
          for ( int x = 0; x < 50; x++) {
            if ( (foldname[x] >= 0x30 && foldname[x] <= 0x39) || foldname[x] == 0x2E) {
            } else {
              if (foldname[x] != 0) foldname[x] = 0x20;
            }
          }

          double i = atof(foldname);
          if ( i > 0 && i < del_number) {
            strcpy (del_numbername, file.name());
            del_number = i;
          }
          //Serial.printf("Name is %s, number is %f\n", foldname, i);
        }
        file = f.openNextFile();

      }
       if (debug) {Serial.printf("lowest is Name is %s, number is %f\n", del_numbername, del_number);}
      if (del_number < 999999999) {
        deleteFolderOrFile(del_numbername);
      }
      full = 1.0 * SD_MMC.usedBytes() / SD_MMC.totalBytes();
       if (debug) {Serial.printf("Disk is %.1f%% full ... \n", 100.0 * full);}
      f.close();
    }
  }
}

//#######################################################################################################    
void deleteFolderOrFile(const char * val) {
  // Function provided by user @gemi254
   if (debug) {Serial.printf("Deleting : %s\n", val);}
  File f = SD_MMC.open(val);
  if (!f) {
     if (debug) {Serial.printf("Failed to open %s\n", val);}
    return;
  }

  if (f.isDirectory()) {
    File file = f.openNextFile();
    while (file) {
      if (file.isDirectory()) {
         if (debug) {Serial.print("  DIR : ");}
         if (debug) {Serial.println(file.name());}
      } else {
         if (debug) {Serial.print("  FILE: ");}
         if (debug) {Serial.print(file.name());}
         if (debug) {Serial.print("  SIZE: ");}
         if (debug) {Serial.print(file.size());}
        if (SD_MMC.remove(file.name())) {
           if (debug) {Serial.println(" deleted.");}
        } else {
           if (debug) {Serial.println(" FAILED.");}
        }
      }
      file = f.openNextFile();
    }
    f.close();
    //Remove the dir
    if (SD_MMC.rmdir(val)) {
       if (debug) {Serial.printf("Dir %s removed\n", val);}
    } else {
       if (debug) {Serial.println("Remove dir failed");}
    }

  } else {
    //Remove the file
    if (SD_MMC.remove(val)) {
       if (debug) {Serial.printf("File %s deleted\n", val);}
    } else {
       if (debug) {Serial.println("Delete failed");}
    }
  }
}

//#######################################################################################################    
//  get_good_jpeg()  - take a picture and make sure it has a good jpeg
//
camera_fb_t *  get_good_jpeg() {
  digitalWrite(camFlashPIN, HIGH);       //Turn the onboard LED on

  camera_fb_t * fb;

  long start;
  int failures = 0;

  do {
    int fblen = 0;
    int foundffd9 = 0;
    long bp = millis();
    long mstart = micros();

    fb = esp_camera_fb_get();
    if (!fb) {
       if (debug) {Serial.println("Camera Capture Failed");}
      failures++;
    } else {
      long mdelay = micros() - mstart;

      int get_fail = 0;

      totalp = totalp + millis() - bp;
      time_in_camera = totalp;

      fblen = fb->len;

      for (int j = 1; j <= 1025; j++) {
        if (fb->buf[fblen - j] != 0xD9) {
          // no d9, try next for
        } else {                                     //Serial.println("Found a D9");
          if (fb->buf[fblen - j - 1] == 0xFF ) {     //Serial.print("Found the FFD9, junk is "); Serial.println(j);
            if (j == 1) {
              normal_jpg++;
            } else {
              extend_jpg++;
            }
            foundffd9 = 1;
            if (debug) {
              if (j > 900) {                             //  rarely happens - sometimes on 2640
                Serial.print("Frame "); Serial.print(frame_cnt); logfile.print("Frame "); logfile.print(frame_cnt);
                Serial.print(", Len = "); Serial.print(fblen); logfile.print(", Len = "); logfile.print(fblen);
                //Serial.print(", Correct Len = "); Serial.print(fblen - j + 1);
                Serial.print(", Extra Bytes = "); Serial.println( j - 1); logfile.print(", Extra Bytes = "); logfile.println( j - 1);
                logfile.flush();
              }

              if ( frame_cnt % 100 == 50) {
                 if (debug) {Serial.printf("Frame %6d, len %6d, extra  %4d, cam time %7d ", frame_cnt, fblen, j - 1, mdelay / 1000);}
                logfile.printf("Frame %6d, len %6d, extra  %4d, cam time %7d ", frame_cnt, fblen, j - 1, mdelay / 1000);
                do_it_now = 1;
              }
            }
            break;
          }
        }
      }

      if (!foundffd9) {
        bad_jpg++;
         if (debug) {Serial.printf("Bad jpeg, Frame %d, Len = %d \n", frame_cnt, fblen);}
        logfile.printf("Bad jpeg, Frame %d, Len = %d\n", frame_cnt, fblen);

        esp_camera_fb_return(fb);
        failures++;

      } else {
        break;
        // count up the useless bytes
      }
    }

  } while (failures < 10);   // normally leave the loop with a break()

  // if we get 10 bad frames in a row, then quality parameters are too high - set them lower (+5), and start new movie
  if (failures == 10) {     
     if (debug) {Serial.printf("10 failures");}
    logfile.printf("10 failures");
    logfile.flush();

    sensor_t * ss = esp_camera_sensor_get();
    int qual = ss->status.quality ;
    ss->set_quality(ss, qual + 5);
    quality = qual + 5;
     if (debug) {Serial.printf("\n\nDecreasing quality due to frame failures %d -> %d\n\n", qual, qual + 5);}
    logfile.printf("\n\nDecreasing quality due to frame failures %d -> %d\n\n", qual, qual + 5);
    delay(1000);

    start_record = 0;
    //reboot_now = true;
  }

  digitalWrite(camFlashPIN, LOW);       //Turn the onboard LED off
  return fb;
}

//#######################################################################################################    
//  eprom functions  - increment the file_group, so files are always unique
//

#include <EEPROM.h>

struct eprom_data {
  int eprom_good;
  int file_group;
};

void do_eprom_read() {

  eprom_data ed;

  EEPROM.begin(200);
  EEPROM.get(0, ed);

  if (ed.eprom_good == MagicNumber) {
     if (debug) {Serial.println("Good settings in the EPROM ");}
    file_group = ed.file_group;
    file_group++;
     if (debug) {Serial.print("New File Group "); Serial.println(file_group );}
  } else {
     if (debug) {Serial.println("No settings in EPROM - Starting with File Group 1 ");}
    file_group = 1;
  }
  do_eprom_write();
  file_number = 1;
}

void do_eprom_write() {

  eprom_data ed;
  ed.eprom_good = MagicNumber;
  ed.file_group  = file_group;

   if (debug) {Serial.println("Writing to EPROM ...");}

  EEPROM.begin(200);
  EEPROM.put(0, ed);
  EEPROM.commit();
  EEPROM.end();
}


//#######################################################################################################    
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
// Make the avi functions
//
//   start_avi() - open the file and write headers
//   another_pic_avi() - write one more frame of movie
//   end_avi() - write the final parameters and close the file


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
// start_avi - open the files and write in headers
//

static esp_err_t start_avi() {

  long start = millis();

   if (debug) {Serial.println("Starting an avi ");}
  char currentDate[50];
  char mon[10];
  char dy[10];
  char yr[10];
  char hr[10];
  char mn[10];
  
  strlcpy(currentDate, itoa(month(), mon, 10), sizeof(currentDate));
  strlcat(currentDate, "_", sizeof(currentDate));
  strlcat(currentDate, itoa(day(), dy, 10), sizeof(currentDate));
  strlcat(currentDate, "_", sizeof(currentDate));
  strlcat(currentDate, itoa(year(), yr, 10), sizeof(currentDate));
  strlcat(currentDate, "_", sizeof(currentDate));
  strlcat(currentDate, itoa(hour(), hr, 10), sizeof(currentDate));
  strlcat(currentDate, ".", sizeof(currentDate));
  strlcat(currentDate, itoa(minute(), mn, 10), sizeof(currentDate));

  //sprintf(avi_file_name, "/%s%d.%03d.avi",  devname, file_group, file_number);  //original
  //sprintf(avi_file_name, "/%s_%s%d.%03d.avi", currentDate, devname, file_group, file_number);
  
  sprintf(avi_file_name, "/%s%d.%03d_(%s).avi", devname, file_group, file_number, currentDate);
  
   if (debug) {Serial.print("DateTime: ");Serial.println(currentDate);}
   if (debug) {Serial.print("File Name: ");Serial.println(avi_file_name);}
   //write to Blynk
   
   terminal.print(F("File: ")); terminal.println(avi_file_name);
   terminal.flush();

  file_number++;

  avifile = SD_MMC.open(avi_file_name, "w");
  idxfile = SD_MMC.open("/idx.tmp", "w");

  if (avifile) {
     if (debug) {Serial.printf("File open: %s\n", avi_file_name);}
    logfile.printf("File open: %s\n", avi_file_name);
  }  else  {
     if (debug) {Serial.println("Could not open file");}
     terminal.println(F("Failed to open AVI file. ERROR!"));
     terminal.flush();
    major_fail();
  }

  if (idxfile)  {
    //Serial.printf("File open: %s\n", "//idx.tmp");
  }  else  {
     if (debug) {Serial.println("Could not open file /idx.tmp");}
     terminal.println(F("Failed to open LOG file."));
     terminal.flush();
    major_fail();
  }

  for ( i = 0; i < AVIOFFSET; i++){
    char ch = pgm_read_byte(&avi_header[i]);
    buf[i] = ch;
  }

  memcpy(buf + 0x40, frameSizeData[framesize].frameWidth, 2);
  memcpy(buf + 0xA8, frameSizeData[framesize].frameWidth, 2);
  memcpy(buf + 0x44, frameSizeData[framesize].frameHeight, 2);
  memcpy(buf + 0xAC, frameSizeData[framesize].frameHeight, 2);

  size_t err = avifile.write(buf, AVIOFFSET);

  avifile.seek( AVIOFFSET, SeekSet);

   if (debug) {Serial.print(F("\nRecording "));}
   if (debug) {Serial.print(avi_length);}
   if (debug) {Serial.println(" seconds.");}

  startms = millis();

  totalp = 0;
  totalw = 0;

  jpeg_size = 0;
  movi_size = 0;
  uVideoLen = 0;
  idx_offset = 4;

  bad_jpg = 0;
  extend_jpg = 0;
  normal_jpg = 0;

  time_in_loop = 0;
  time_in_camera = 0;
  time_in_sd = 0;
  time_in_good = 0;
  time_total = 0;
  time_in_web1 = 0;
  time_in_web2 = 0;
  delay_wait_for_sd = 0;
  wait_for_cam = 0;

  time_in_sd += (millis() - start);

  logfile.flush();
  avifile.flush();

} // end of start avi


//#######################################################################################################    
//  another_save_avi saves another frame to the avi file, uodates index
//           -- pass in a fb pointer to the frame to add
//

static esp_err_t another_save_avi(camera_fb_t * fb ) {

  long start = millis();

  int fblen;
  fblen = fb->len;

  int fb_block_length;
  uint8_t* fb_block_start;

  jpeg_size = fblen;

  remnant = (4 - (jpeg_size & 0x00000003)) & 0x00000003;

  long bw = millis();
  long frame_write_start = millis();

  framebuffer_static[0] = 0x30;       // "00dc"
  framebuffer_static[1] = 0x30;
  framebuffer_static[2] = 0x64;
  framebuffer_static[3] = 0x63;

  int jpeg_size_rem = jpeg_size + remnant;

  framebuffer_static[4] = jpeg_size_rem % 0x100;
  framebuffer_static[5] = (jpeg_size_rem >> 8) % 0x100;
  framebuffer_static[6] = (jpeg_size_rem >> 16) % 0x100;
  framebuffer_static[7] = (jpeg_size_rem >> 24) % 0x100;

  fb_block_start = fb->buf;

  if (fblen > fbs * 1024 - 8 ) {                     // fbs is the size of frame buffer static
    fb_block_length = fbs * 1024;
    fblen = fblen - (fbs * 1024 - 8);
    memcpy(framebuffer_static + 8, fb_block_start, fb_block_length - 8);
    fb_block_start = fb_block_start + fb_block_length - 8;

  } else {
    fb_block_length = fblen + 8  + remnant;
    memcpy(framebuffer_static + 8, fb_block_start,  fblen);
    fblen = 0;
  }

  size_t err = avifile.write(framebuffer_static, fb_block_length);

  if (err != fb_block_length) {
     if (debug) {Serial.print("Error on avi write: err = "); Serial.print(err);}
     if (debug) {Serial.print(" len = "); Serial.println(fb_block_length);}
    logfile.print("Error on avi write: err = "); logfile.print(err);
    logfile.print(" len = "); logfile.println(fb_block_length);
    terminal.print(F("Error on AVI write. Error="));terminal.println(err);
    terminal.flush();
  }

  while (fblen > 0) {

    if (fblen > fbs * 1024) {
      fb_block_length = fbs * 1024;
      fblen = fblen - fb_block_length;
    } else {
      fb_block_length = fblen  + remnant;
      fblen = 0;
    }

    memcpy(framebuffer_static, fb_block_start, fb_block_length);

    size_t err = avifile.write(framebuffer_static,  fb_block_length);

    if (err != fb_block_length) {
       if (debug) {Serial.print("Error on avi write: err = "); Serial.print(err);}
       if (debug) {Serial.print(" len = "); Serial.println(fb_block_length);}
       terminal.print(F("Error on AVI write. Error="));terminal.println(err);
       terminal.flush();
    }

    fb_block_start = fb_block_start + fb_block_length;
    delay(0);
  }


  movi_size += jpeg_size;
  uVideoLen += jpeg_size;
  long frame_write_end = millis();

  print_2quartet(idx_offset, jpeg_size, idxfile);

  idx_offset = idx_offset + jpeg_size + remnant + 8;

  movi_size = movi_size + remnant;

  if ( do_it_now == 1) {
    do_it_now = 0;
     if (debug) {Serial.printf(" sd time %4d -- \n",  millis() - bw);}
    logfile.printf(" sd time %4d -- \n",  millis() - bw);
    logfile.flush();
  }

  totalw = totalw + millis() - bw;
  time_in_sd += (millis() - start);

  avifile.flush();


} // end of another_pic_avi

//#######################################################################################################    
//  end_avi writes the index, and closes the files
//

static esp_err_t end_avi() {

  long start = millis();

  unsigned long current_end = avifile.position();

   if (debug) {Serial.println("End of avi - closing the files");}
  logfile.println("End of avi - closing the files");

  if (frame_cnt <  5 ) {
     if (debug) {Serial.println("Recording screwed up, less than 5 frames, forget index\n");}
     terminal.println(F("Recording messed up. Less than 5 frames, forget index."));
     terminal.flush();
     idxfile.close();
     avifile.close();
     int xx = remove("/idx.tmp");
     int yy = remove(avi_file_name);

  } else {

    elapsedms = millis() - startms;

    float fRealFPS = (1000.0f * (float)frame_cnt) / ((float)elapsedms) * speed_up_factor;

    float fmicroseconds_per_frame = 1000000.0f / fRealFPS;
    uint8_t iAttainedFPS = round(fRealFPS) ;
    uint32_t us_per_frame = round(fmicroseconds_per_frame);

    //Modify the MJPEG header from the beginning of the file, overwriting various placeholders

    avifile.seek( 4 , SeekSet);
    print_quartet(movi_size + 240 + 16 * frame_cnt + 8 * frame_cnt, avifile);

    avifile.seek( 0x20 , SeekSet);
    print_quartet(us_per_frame, avifile);

    unsigned long max_bytes_per_sec = (1.0f * movi_size * iAttainedFPS) / frame_cnt;

    avifile.seek( 0x24 , SeekSet);
    print_quartet(max_bytes_per_sec, avifile);

    avifile.seek( 0x30 , SeekSet);
    print_quartet(frame_cnt, avifile);

    avifile.seek( 0x8c , SeekSet);
    print_quartet(frame_cnt, avifile);

    avifile.seek( 0x84 , SeekSet);
    print_quartet((int)iAttainedFPS, avifile);

    avifile.seek( 0xe8 , SeekSet);
    print_quartet(movi_size + frame_cnt * 8 + 4, avifile);

    if (debug)
    {
      Serial.println(F("\n*** Video recorded and saved ***\n"));
      Serial.printf("Recorded %5d frames in %5d seconds\n", frame_cnt, elapsedms / 1000);
      Serial.printf("File size is %u bytes\n", movi_size + 12 * frame_cnt + 4);
      Serial.printf("Adjusted FPS is %5.2f\n", fRealFPS);
      Serial.printf("Max data rate is %lu bytes/s\n", max_bytes_per_sec);
      Serial.printf("Frame duration is %d us\n", us_per_frame);
      Serial.printf("Average frame length is %d bytes\n", uVideoLen / frame_cnt);
      Serial.print("Average picture time (ms) "); Serial.println( 1.0 * totalp / frame_cnt);
      Serial.print("Average write time (ms)   "); Serial.println( 1.0 * totalw / frame_cnt );
      Serial.print("Normal jpg % ");  Serial.println( 100.0 * normal_jpg / frame_cnt, 1 );
      Serial.print("Extend jpg % ");  Serial.println( 100.0 * extend_jpg / frame_cnt, 1 );
      Serial.print("Bad    jpg % ");  Serial.println( 100.0 * bad_jpg / frame_cnt, 5 );
      Serial.printf("Writng the index, %d frames\n", frame_cnt);
    }

    logfile.printf("Recorded %5d frames in %5d seconds\n", frame_cnt, elapsedms / 1000);
    logfile.printf("File size is %u bytes\n", movi_size + 12 * frame_cnt + 4);
    logfile.printf("Adjusted FPS is %5.2f\n", fRealFPS);
    logfile.printf("Max data rate is %lu bytes/s\n", max_bytes_per_sec);
    logfile.printf("Frame duration is %d us\n", us_per_frame);
    logfile.printf("Average frame length is %d bytes\n", uVideoLen / frame_cnt);
    logfile.print("Average picture time (ms) "); logfile.println( 1.0 * totalp / frame_cnt);
    logfile.print("Average write time (ms)   "); logfile.println( 1.0 * totalw / frame_cnt );
    logfile.print("Normal jpg % ");  logfile.println( 100.0 * normal_jpg / frame_cnt, 1 );
    logfile.print("Extend jpg % ");  logfile.println( 100.0 * extend_jpg / frame_cnt, 1 );
    logfile.print("Bad    jpg % ");  logfile.println( 100.0 * bad_jpg / frame_cnt, 5 );

    logfile.printf("Writng the index, %d frames\n", frame_cnt);

    avifile.seek( current_end , SeekSet);

    idxfile.close();

    size_t i1_err = avifile.write(idx1_buf, 4);

    print_quartet(frame_cnt * 16, avifile);

    idxfile = SD_MMC.open("/idx.tmp", "r");

    if (idxfile)  {
      //Serial.printf("File open: %s\n", "//idx.tmp");
      //logfile.printf("File open: %s\n", "/idx.tmp");
    }  else  {
       if (debug) {Serial.println("Could not open index file");}
      logfile.println("Could not open index file");
      major_fail();
    }

    char * AteBytes;
    AteBytes = (char*) malloc (8);

    for (int i = 0; i < frame_cnt; i++) {
      size_t res = idxfile.readBytes( AteBytes, 8);
      size_t i1_err = avifile.write(dc_buf, 4);
      size_t i2_err = avifile.write(zero_buf, 4);
      size_t i3_err = avifile.write((uint8_t *)AteBytes, 8);
    }

    free(AteBytes);

    idxfile.close();
    avifile.close();

    int xx = remove("/idx.tmp");
  }

   if (debug) {Serial.println("---");  logfile.println("---");}

  time_in_sd += (millis() - start);

   if (debug) {Serial.println("");}
  time_total = millis() - startms;
  if (debug)
  {
    Serial.printf("waiting for cam %10dms, %4.1f%%\n", wait_for_cam , 100.0 * wait_for_cam  / time_total);
    Serial.printf("Time in camera  %10dms, %4.1f%%\n", time_in_camera, 100.0 * time_in_camera / time_total);
    Serial.printf("waiting for sd  %10dms, %4.1f%%\n", delay_wait_for_sd , 100.0 * delay_wait_for_sd  / time_total);
    Serial.printf("Time in sd      %10dms, %4.1f%%\n", time_in_sd    , 100.0 * time_in_sd     / time_total);
    Serial.printf("web (core 1)    %10dms, %4.1f%%\n", time_in_web1  , 100.0 * time_in_web1   / time_total);
    Serial.printf("web (core 0)    %10dms, %4.1f%%\n", time_in_web2  , 100.0 * time_in_web2   / time_total);
    Serial.printf("time total      %10dms, %4.1f%%\n", time_total    , 100.0 * time_total     / time_total);
  }

  logfile.printf("waiting for cam %10dms, %4.1f%%\n", wait_for_cam , 100.0 * wait_for_cam  / time_total);
  logfile.printf("Time in camera  %10dms, %4.1f%%\n", time_in_camera, 100.0 * time_in_camera / time_total);
  logfile.printf("waiting for sd  %10dms, %4.1f%%\n", delay_wait_for_sd , 100.0 * delay_wait_for_sd  / time_total);
  logfile.printf("Time in sd      %10dms, %4.1f%%\n", time_in_sd    , 100.0 * time_in_sd     / time_total);
  logfile.printf("web (core 1)    %10dms, %4.1f%%\n", time_in_web1  , 100.0 * time_in_web1   / time_total);
  logfile.printf("web (core 0)    %10dms, %4.1f%%\n", time_in_web2  , 100.0 * time_in_web2   / time_total);
  logfile.printf("time total      %10dms, %4.1f%%\n", time_total    , 100.0 * time_total     / time_total);

  logfile.flush();

  //frame_cnt = 0;  //DJA need to reset since Blynk can be used to start another movie.

}

void the_camera_loop (void* pvParameter);
void the_sd_loop (void* pvParameter);
void delete_old_stuff();

//#######################################################################################################    


BLYNK_CONNECTED() {
  // Synchronize time on connection
  rtc.begin();
  Blynk.virtualWrite(V1,0);   //set the button back to Start if not already done
  Blynk.virtualWrite(V2,0);   //set the SD card space to 0
  Blynk.virtualWrite(V3,0);   //set the delay start slider back to default
  //Blynk.setProperty(V1, "label", "Start");
  //Blynk.setProperty(V1, "onColor", "Start");
  //Blynk.setProperty(V1, "offColor", "Start");
  //Blynk.setProperty(V1, "onBackColor", "Start");
  //Blynk.setProperty(V1, "offBackColor", "Start");

}


BLYNK_WRITE(V1) //Delayed Start Button (on or off)
{
  delay_start = param.asInt();
  if(delay_start)
  {
    //delayStartTimer.restartTimer(dsTimer);  //enable the timer
    //delayStartTimer.enable(dsTimer);        //restart it
    dsTimer = delayStartTimer.setTimeout(record_delay_start, doStartRecord);
    //frame_cnt = 0;  //if > 1 round in, have to reset this.

    //if(delayStartTimer.isEnabled(dsTimer)) {terminal.println(F("Timer is enabled...")); }
    terminal.print(F("Delayed recording starting in "));terminal.print(record_delay_start/60/1000);terminal.println(F(" minute(s)."));terminal.flush();
    //terminal.print(F("Frame Count check "));terminal.println(frame_cnt);terminal.flush();
  } else {
     delayStartTimer.disable(dsTimer);
     start_record = 0;              //stops recording.
     terminal.println(F("Recording was stopped from Blynk App."));terminal.flush();  //clicked the Stop button in Blynk
  }
}

BLYNK_WRITE(V3) //Slider Delay Start Time (minutes)
{
  if (!delay_start)  //can only change the time if not started
  {
    record_delay_start = param.asInt() * 60 * 1000; // param val returned is minutes, then convert to ms.
    if (record_delay_start == 0) {record_delay_start = 1000;}  //just make it a second - start right away
    //default set in program:  record_delay_start = 20 * 60 * 1000;  
    terminal.print(F("Delay time changed to "));terminal.print(param.asInt());terminal.println(F(" minute(s)."));terminal.flush();
    //dsTimer = delayStartTimer.setTimeout(record_delay_start, doStartRecord);
    //delayStartTimer.restartTimer(dsTimer);
    //delayStartTimer.disable(dsTimer);  //dont start it as this is just changing time
  }
  else
  {
    terminal.println(F("You can't change the delay once recording has started."));
  }
}

void doStartRecord()    // called when timer delayStartTimer counts down
{
    start_record = 1;
    delayStartTimer.disable(dsTimer); //disable the timer so it doesn't keep triggering after x time
    //Blynk.virtualWrite(V1,1);   //set the button to stop as recording started on timer
    terminal.println(F("Recording started from timer."));terminal.flush();
    if (debug) {Serial.println("doStartRecord() TRIGGERED!!!!!!!!!!!!!!!!");}
}


//#######################################################################################################    
//#######################################################################################################    
void setup() {
  WRITE_PERI_REG(RTC_CNTL_BROWN_OUT_REG, 0);  // Turn-off the 'brownout detector'

  Serial.begin(115200);
   if (debug) {Serial.println("\n\n---");}
   
  LDRSenReading.begin(); //moving average

  Blynk.begin(auth, ssid, pass);
  delay(1000);
  if (!Blynk.connected()){Serial.println("BLYNK DID NOT CONNECT!");}
  if (debug) {Serial.print("IP Address:");Serial.println(WiFi.localIP());}
  terminal.clear();   //clear the Blynk terminal window to start a fresh recording
  terminal.println(F("RLapseCam connected and reporting for duty!"));terminal.println(F(""))
  terminal.print(F("IP Address: "));terminal.println(WiFi.localIP());
  terminal.print(F("Date: "));terminal.print(month());terminal.print(F("/"));terminal.print(day());terminal.print(F("/"));terminal.print(year());
  terminal.print(F(" Time: "));terminal.print(hour());terminal.print(F(":"));terminal.println(minute());
  terminal.flush();
  

  pinMode(LDRSensorPIN, INPUT);
  pinMode(camFlashPIN, OUTPUT);               // Attention - shared with the LDR sensor, which is ADC input.  Have to set mode with each use of the LED
  digitalWrite(camFlashPIN, LOW);            //Turn the onboard LED on
  //pinMode(4, INPUT);                //See above

  //pinMode(12, INPUT_PULLUP);        // pull this down to stop recording
  //pinMode(13, INPUT_PULLUP);        // pull this down switch wifi

  //pinMode (camFlashPIN, OUTPUT);//Specify that LED pin is output
  //digitalWrite(camFlashPIN, HIGH);  //Turn the onboard LED on

  
  //Serial.setDebugOutput(true);

  if (debug)
  {
    Serial.println("                                    ");
    Serial.println("-------------------------------------");
    Serial.printf("ESP32-CAM-Video-Recorder-junior %s\n", vernum);
    Serial.println("-------------------------------------");
    Serial.print("setup, core ");  Serial.print(xPortGetCoreID());
    Serial.print(", priority = "); Serial.println(uxTaskPriorityGet(NULL));
  }

  esp_reset_reason_t reason = esp_reset_reason();

  logfile.print("--- reboot ------ because: ");
   if (debug) {Serial.print("--- reboot ------ because: ");}
  //terminal.print(F("!!!!!  Something went wrong. Reason Code:  "));terminal.print(reason);terminal.println(F("  !!!!!"));
  //terminal.flush();
  switch (reason) {
    case ESP_RST_UNKNOWN : Serial.println("ESP_RST_UNKNOWN"); logfile.println("ESP_RST_UNKNOWN"); break;
    case ESP_RST_POWERON : Serial.println("ESP_RST_POWERON"); logfile.println("ESP_RST_POWERON"); break;
    case ESP_RST_EXT : Serial.println("ESP_RST_EXT"); logfile.println("ESP_RST_EXT"); break;
    case ESP_RST_SW : Serial.println("ESP_RST_SW"); logfile.println("ESP_RST_SW"); break;
    case ESP_RST_PANIC : Serial.println("ESP_RST_PANIC"); logfile.println("ESP_RST_PANIC"); break;
    case ESP_RST_INT_WDT : Serial.println("ESP_RST_INT_WDT"); logfile.println("ESP_RST_INT_WDT"); break;
    case ESP_RST_TASK_WDT : Serial.println("ESP_RST_TASK_WDT"); logfile.println("ESP_RST_TASK_WDT"); break;
    case ESP_RST_WDT : Serial.println("ESP_RST_WDT"); logfile.println("ESP_RST_WDT"); break;
    case ESP_RST_DEEPSLEEP : Serial.println("ESP_RST_DEEPSLEEP"); logfile.println("ESP_RST_DEEPSLEEP"); break;
    case ESP_RST_BROWNOUT : Serial.println("ESP_RST_BROWNOUT"); logfile.println("ESP_RST_BROWNOUT"); break;
    case ESP_RST_SDIO : Serial.println("ESP_RST_SDIO"); logfile.println("ESP_RST_SDIO"); break;
    default  : Serial.println("Reset resaon"); logfile.println("ESP ???"); break;
  }

  //Serial.printf("Internal Total heap %d, internal Free Heap %d\n", ESP.getHeapSize(), ESP.getFreeHeap());
  //Serial.printf("SPIRam Total heap   %d, SPIRam Free Heap   %d\n", ESP.getPsramSize(), ESP.getFreePsram());

   if (debug) {Serial.println("Reading the eprom  ...");}
  do_eprom_read();

  // SD camera init
   if (debug) {Serial.println("Mounting the SD card ...");}
   terminal.println(F("Mounting the SD card..."));
   terminal.flush();   
  esp_err_t card_err = init_sdcard();
  if (card_err != ESP_OK) {
     if (debug) {Serial.printf("SD Card init failed with error 0x%x", card_err);}
     terminal.println(F("Failed to mount SD card! Make sure it's there and working."));
     terminal.flush();
    major_fail();
    return;
  }

  float SDSpace = 100.0 * SD_MMC.usedBytes() / SD_MMC.totalBytes();
  terminal.print(F("SD Card usage:  "));terminal.print(SDSpace);terminal.println(F("%"));
  terminal.flush();

  //Update the SD card bar on Blynk
  Blynk.virtualWrite(V2,int(SDSpace));   //set the SD card state

  devstr.toCharArray(devname, devstr.length());          // name of your camera for mDNS, Router, and filenames

   if (debug) {Serial.println("Try to get parameters from config.txt ...");}
  read_config_file();


  char logname[50];
  sprintf(logname, "/%s%d.999.txt",  devname, file_group);
   if (debug) {Serial.printf("Creating logfile %s\n",  logname);}
  logfile = SD_MMC.open(logname, FILE_WRITE);

  if (!logfile) {
     if (debug) {Serial.println("Failed to open logfile for writing");}
  }
//  if (IncludeInternet > 0) {
//    Serial.println("Starting the wifi ...");
//    init_wifi();
//    InternetOff = false;
//  }
  
  terminal.println(F("Checking camera status..."));
  terminal.flush();
  
   if (debug) {Serial.println("Setting up the camera...");}
  config_camera();

   if (debug) {Serial.println("Checking SD for available space...");}
   terminal.println(F("Checking SD for available space..."));
   terminal.flush();
  delete_old_stuff();
 
  //digitalWrite(33, HIGH);         // red light turns off when setup is complete

/*
  if ( !InternetOff && IncludeInternet == 1) {
    Serial.printf("Shutting off WiFi now \n\n");
    delay(1000);
    WiFi.disconnect();
    InternetOff = true;
  }


  if ( !InternetOff && IncludeInternet > 1) {
    Serial.println("Starting Web Services ...");
    startCameraServer();
  }
*/

  framebuffer = (uint8_t*)ps_malloc(1024 * 1024); // buffer to store a jpg in motion // needs to be larger for big frames from ov5640 

   if (debug) {Serial.println("Creating the_camera_loop_task");}

  wait_for_sd = xSemaphoreCreateMutex();
  sd_go = xSemaphoreCreateMutex();

  xSemaphoreTake( wait_for_sd, portMAX_DELAY );   // will be "given" when sd write is done
  xSemaphoreTake( sd_go, portMAX_DELAY );         // will be "given" when sd write should start

  // prio 3 - higher than the camera loop(), and the streaming
  xTaskCreatePinnedToCore( the_camera_loop, "the_camera_loop", 3000, NULL, 3, &the_camera_loop_task, 0); // prio 3, core 0

  delay(100);

  // prio 4 - higher than the cam_loop(), and the streaming
  xTaskCreatePinnedToCore( the_sd_loop, "the_sd_loop", 2000, NULL, 4, &the_sd_loop_task, 1);  // prio 4, core 1

  delay(200);

  boot_time = millis();
  //const char *strdate = ctime(&now);
  String strdate = String(hour()) + ":" + minute() + ":" + second() + " " + day() + " " + month() + " " + year();
  //String currentDate = String(day()) + " " + month() + " " + year();
  
  logfile.println(strdate);

  terminal.println(F("I'm ready! Let's do this!"));
  terminal.println(F(""));
  terminal.flush();
  terminal.print(F("Instructions: Select delay start and click Start. Recording will stop after the following conditions:  "));
  terminal.print(avi_length);
  terminal.print(F(" seconds, or you clicked Stop, or the print is idle for "));
  terminal.print(time_to_wait_till_print_done/1000);
  terminal.println(F(" seconds or more."));
  terminal.flush();

  //dsTimer = delayStartTimer.setTimeout(record_delay_start, doStartRecord);   //setup the timer
  //delayStartTimer.disable(dsTimer);  //dont start it yet. 

  //Enable the photo webserver for taking photos
  // Route for root / web page
  photoServer.on("/", HTTP_GET, [](AsyncWebServerRequest * request) {
    request->send_P(200, "text/html", index_html);
  });

  photoServer.on("/capture", HTTP_GET, [](AsyncWebServerRequest * request) {
    takeNewPhoto = true;
    request->send_P(200, "text/plain", "Taking Photo");
  });

  photoServer.on("/saved-photo", HTTP_GET, [](AsyncWebServerRequest * request) {
    request->send(SD_MMC, FILE_PHOTO, "image/jpg", false);
  });

  
  photoServer.begin();  // Start photo server for web page img capture viewing

  // if (debug) {Serial.println("  End of setup()\n\n");}
}

//#######################################################################################################    
// the_sd_loop()
void the_sd_loop (void* pvParameter) {

   if (debug) {Serial.print("the_sd_loop, core ");  Serial.print(xPortGetCoreID());}
   if (debug) {Serial.print(", priority = "); Serial.println(uxTaskPriorityGet(NULL));}

  while (1) {
    xSemaphoreTake( sd_go, portMAX_DELAY );            // we wait for camera loop to tell us to go
    another_save_avi( fb_curr);                        // do the actual sd wrte
    xSemaphoreGive( wait_for_sd );                     // tell camera loop we are done
  }
}

//#######################################################################################################    
// the_camera_loop()  - this runs in paralell to the main loop() task on the other CPU - has higher priority
void the_camera_loop (void* pvParameter) {

   if (debug) {Serial.print("the loop, core ");  Serial.print(xPortGetCoreID());}
   if (debug) {Serial.print(", priority = "); Serial.println(uxTaskPriorityGet(NULL));}

  frame_cnt = 0;
  //start_record_2nd_opinion = digitalRead(12);
  //start_record_1st_opinion = digitalRead(12);
  //start_record = 0;

  delay(500);

  while (1) {
    // if (frame_cnt == 0 && start_record == 0)  // do nothing
    // if (frame_cnt == 0 && start_record == 1)  // start a movie
    // if (frame_cnt > 0 && start_record == 0)   // stop the movie
    // if (frame_cnt > 0 && start_record != 0)   // another frame

    Blynk.run();
    delayStartTimer.run();
    if(delayStartTimer.isEnabled(dsTimer)) {current_frame_time = millis();}   //make sure we dont trigger the no activity movie closure
    delay(10);
    //collect and average readings from the LDR sensor
    pc = analogRead(LDRSensorPIN); // read the photocell
    avgLDR = LDRSenReading.reading(pc);    // calculate the moving average

    //if (debug) {Serial.print("In MAIN CAM LOOP: ");Serial.print("FrameCNT=");Serial.print(frame_cnt);;Serial.print(" start_record=");Serial.println(start_record);}
    
  
    ///////////////////  NOTHING TO DO //////////////////
    if (frame_cnt == 0 && start_record == 0) {
      if (we_are_already_stopped == 0) 
       {
         if (debug) {Serial.println("\n\nRecording has either not started or has been stopped and is complete.\n\n");}
        //terminal.println(F("Recording has completed and has stopped."));
        //terminal.flush();      
        //Blynk.virtualWrite(V1,0);   //set the button back to Start
        we_are_already_stopped = 1;
        delay(100);
       }

      ///////////////////  START A MOVIE  //////////////////
      //This sets up the AVI file to record frames. Done once per new movie
    } else if (frame_cnt == 0 && start_record == 1) {

      terminal.println(F(""));
      terminal.print(F("Recording started: "));
      terminal.print(month());terminal.print(F("/"));terminal.print(day());terminal.print(F("/"));terminal.print(year());
      terminal.print(F(" @ "));terminal.print(hour());terminal.print(F(":"));terminal.println(minute());
      terminal.flush();      
      terminal.flush();

      we_are_already_stopped = 0;

      delete_old_stuff();

      avi_start_time = millis();
       if (debug) {Serial.printf("\nStart the avi ... at %d\n", avi_start_time);}
       if (debug) {Serial.printf("Framesize %d, quality %d, length %d seconds\n\n", framesize, quality, avi_length);}
      logfile.printf("\nStart the avi ... at %d\n", avi_start_time);
      logfile.printf("Framesize %d, quality %d, length %d seconds\n\n", framesize, quality, avi_length);
      logfile.flush();

      frame_cnt++;

      long wait_for_cam_start = millis();
      fb_curr = get_good_jpeg();                     // should take zero time
      wait_for_cam += millis() - wait_for_cam_start;

      start_avi();

      wait_for_cam_start = millis();

      fb_next = get_good_jpeg();                    // should take nearly zero time due to time spent writing header
      wait_for_cam += millis() - wait_for_cam_start;

      xSemaphoreGive( sd_go );                     // trigger sd write to write first frame

      //digitalWrite(33, frame_cnt % 2);                // blink

      ///////////////////  END THE MOVIE //////////////////
    } else if ( (frame_cnt > 0 && start_record == 0) ||  millis() > (avi_start_time + avi_length * 1000)) { // end the avi

       if (debug) {Serial.println("End the Avi");}
      terminal.println(F("Video ended & saved!"));
      terminal.flush();
      Blynk.virtualWrite(V1,0);   //set the button back to Start


      xSemaphoreTake( wait_for_sd, portMAX_DELAY );
      esp_camera_fb_return(fb_curr);

      frame_cnt++;
      fb_curr = fb_next;
      fb_next = NULL;

      xSemaphoreGive( sd_go );                  // save final frame of movie

      //digitalWrite(33, frame_cnt % 2);

      xSemaphoreTake( wait_for_sd, portMAX_DELAY );    // wait for final frame of movie to be written

      esp_camera_fb_return(fb_curr);
      fb_curr = NULL;

      end_avi();                                // end the movie

      //digitalWrite(33, HIGH);          // light off

      avi_end_time = millis();

      float fps = 1.0 * frame_cnt / ((avi_end_time - avi_start_time) / 1000) ;

       if (debug) {Serial.printf("End the avi at %d.  It was %d frames, %d ms at %.2f fps...\n", millis(), frame_cnt, avi_end_time, avi_end_time - avi_start_time, fps);}
      logfile.printf("End the avi at %d.  It was %d frames, %d ms at %.2f fps...\n", millis(), frame_cnt, avi_end_time, avi_end_time - avi_start_time, fps);

      frame_cnt = 0;             // start recording again on the next loop

      ///////////////////  ANOTHER FRAME  //////////////////
    } else if (frame_cnt > 0 && start_record != 0) {  // another frame of the avi

      /*DJA - Modified.  How this works.  Original code looped based on frame count
       * Intent was to jump into that and only take a single frame and add it to the AVI
       * when the LDR went from a dark to light (UV on) state. Then wait until it again
       * went to a light on state.  The While loop waits while dark, when done it breaks 
       * out and takes the first frame, then frameCapture is set to True so subsequent
       * frames are not taken until it goes dark, then light again (into then out of the while loop)
      */

      //make sure we dont trigger the no activity movie closure
      if(delayStartTimer.isEnabled(dsTimer)) {current_frame_time = millis();}   

      if(avgLDR < 1000)  //UV LEDs off - so don't take any pictures.  
      {
        frameCapture = false;
        delay(10);  //feed the watchdog

        if (debug) 
        {
            Serial.print("Counting down: Millis:  ");Serial.print(millis()); 
            Serial.print(" Time to wait: ");Serial.println(time_to_wait_till_print_done + current_frame_time);
        }       
        
        // end movie if nothing for 120 seconds - most pauses between layers is max 30-60 seconds (1st layers)
        if (millis() > current_frame_time + time_to_wait_till_print_done) { 
           if (debug) {Serial.print("Recording stopped after ");Serial.print(time_to_wait_till_print_done);Serial.print(" seconds due to inactivity.");}
           terminal.println(F("Recording stopped due to inactivity. Print must be done?"));
           terminal.flush();   
           Blynk.virtualWrite(V1,0);   //set the button back to Start        
           start_record = 0; // start_record is the global to start(1) or stop(0) recording
           //break;  //break out of the loop and don't start another recording.
        }
        //Stopped from Blynk, break out of this...
        if (start_record == 0) {Blynk.virtualWrite(V1,0);}  
      }
      else  //when LDR goes high (UV light comes on) capture a frame. 
      {
   
        if (!frameCapture)   //DJA - only do it once while UV light is on
        {
          current_frame_time = millis();  //reset so recording does not stop based on above check

          frame_cnt++;  

          //update on Blynk terminal 

          //First frames < 10 are critical to ensure all is good, so print a notice with each frame taken
          if (frame_cnt < 10) {  
            terminal.print(F("Frame captured. #"));
            terminal.println(frame_cnt);
            terminal.flush();
          }        
        
          if ((frame_cnt % 10) == 0 ) {  //every 10 frames.  
            if (frame_cnt == 10)
            {
              terminal.print(F("Frame captured. #"));
              terminal.println(frame_cnt);             
              terminal.println(F(" (Rest will report every 10 frames)")); 
            } else {
              terminal.print(F("Frame captured. #"));
              terminal.println(frame_cnt);
            }
            terminal.flush();
          }        

          long delay_wait_for_sd_start = millis();
          xSemaphoreTake( wait_for_sd, portMAX_DELAY );             // make sure sd writer is done
          delay_wait_for_sd += millis() - delay_wait_for_sd_start;

          esp_camera_fb_return(fb_curr);

          fb_curr = fb_next;           // we will write a frame, and get the camera preparing a new one

          xSemaphoreGive( sd_go );             // write the frame in fb_curr

          long wait_for_cam_start = millis();
          fb_next = get_good_jpeg();               // should take near zero, unless the sd is faster than the camera, when we will have to wait for the camera
          wait_for_cam += millis() - wait_for_cam_start;

          //digitalWrite(33, frame_cnt % 2);
          if (debug) {Serial.println("Another frame");}  
          frameCapture = true;              //DJA
        }
        delay(10);  //Feed the WDT


        if (frame_cnt % 100 == 10 ) {     // print some status every 100 frames
          if (frame_cnt == 10) {
            bytes_before_last_100_frames = movi_size;
            time_before_last_100_frames = millis();
            most_recent_fps = 0;
            most_recent_avg_framesize = 0;
            //log results to Blynk Terminal
 
          
          } else {

            most_recent_fps = 100.0 / ((millis() - time_before_last_100_frames) / 1000.0) ;
            most_recent_avg_framesize = (movi_size - bytes_before_last_100_frames) / 100;


            if (Lots_of_Stats) {
              if (debug) {Serial.printf("So far: %04d frames, in %6.1f seconds, for last 100 frames: avg frame size %6.1f kb, %.2f fps ...\n", frame_cnt, 0.001 * (millis() - avi_start_time), 1.0 / 1024  * most_recent_avg_framesize, most_recent_fps);}
              logfile.printf("So far: %04d frames, in %6.1f seconds, for last 100 frames: avg frame size %6.1f kb, %.2f fps ...\n", frame_cnt, 0.001 * (millis() - avi_start_time), 1.0 / 1024  * most_recent_avg_framesize, most_recent_fps);
            }

            total_delay = 0;

            bytes_before_last_100_frames = movi_size;
            time_before_last_100_frames = millis();
          }
        }
      }
    }  //end ANOTHER FRAME
  }
  
}

//#######################################################################################################    
// loop() - loop runs at low prio, so I had to move it to the task the_camera_loop at higher priority

void loop() {
  long run_time = millis() - boot_time;
  //Blynk.run();

  //pc = analogRead(LDRSensorPIN); // read the photocell
  //avgLDR = LDRSenReading.reading(pc);    // calculate the moving average
  //Serial.print("(main loop) LDR Value: ");  Serial.println(avgLDR);
  //sensorValue = analogRead(LDRSensor);

  if (takeNewPhoto) 
  {
     capturePhotoSaveToSD();
     takeNewPhoto = false;
  }


  //if (reboot_now == true) {
  //  delay(2000);
  //  major_fail();
  //}

  delay(20); //was 200  Note - any faster and issues caused in camera loop 20ms is good. 

}
