
/*
  Credit:  GoProControl:  https://github.com/aster94/GoProControl
  Control your GoPro with the Serial Monitor (etc)
  edit the file Secrets.h with your camera netword name and password
  CAMERA could be: HERO3, HERO4, HERO5, HERO6, HERO7, FUSION, HERO8, MAX

  Fonts: 
  https://fonts.google.com/specimen/Orbitron#standard-styles orbitron font
  Used to convet font to GFX fonts:  https://rop.nl/truetype2gfx/
  Multiple fonts include but not all used.  See includes - can remove others if not being used

  Libraries Used: GoProControl, Moving Average, Tone32, M5StickCPlus. These can all be found either 
  in the library mgr or go directly to Github and search for them and add them to your environment

  
  *** I M P O R T A N T ***
  This has been tested on the GoPro Hero 7 Black.  However, looking at the Git repo, it should work
  with others.  This uses the M5Stick-C Plus. Also tested with M5Stack Atom Lite and should work with other
  ESP boards. 
  
  There's an issue somewhere between comms of the ESP and GoPro.  This took me a week to figure out.  Turns out
  that if you disconnect / power down the ESP or GoPro without first successfully calling gp.end() something 
  gets hung in the comms.  The ESP will still connect to the GoPro but it will return connection lost errors. 
  To fix this, simply disconnect ESP power, turn off GoPro and remove battery (I did for a few mins, but it may  take less)
  Then power up the GoPro, then the ESP and try again. I think it also works if you just remove the GoPor
  battery, wait a bit, then power it back on and re-connect.  

  Warranty:  This software is distributed WITHOUT ANY WARRANTY, even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
  for more details.

v9 - added frameskip option and changed what buttons do
v8 - streamlining and cleaning up LCD display functions, adding battery monitoring
v7 - adding built-in buzzer
v6 - integrating LDR sensor and sensing
v5 - adding fonts
v4 - testing with M5Stick-C Plus 
*/

//#include "M5Atom.h"       //If using M5Atom
#include <M5StickCPlus.h>
#include "AXP192.h"         //M5Stick-C Plus
#include <GoProControl.h>
#include "Secrets.h"
#include <movingAvg.h> 
#include <Tone32.h>         //M5Stick-C Plus
#include "Orbitron_ExtraBold9pt7b.h"
#include "Orbitron_ExtraBold12pt7b.h" 
#include "Orbitron_ExtraBold14pt7b.h" 
//#include "Orbitron_ExtraBold16pt7b.h"
//#include "Orbitron_ExtraBold18pt7b.h" 

//##############################################################################
// D E B U G  - set True to turn on Serial Debugging
#define debug false
//##############################################################################

#define LDRSensorPIN 33    //was 26 but now working??
#define BUZZER_PIN 0
#define BUZZER_CHANNEL 0

#define CAMERA HERO7 // Change here for your camera
//from my Hero 7:  06 41 69 EF 79 94
byte remote_MAC_ADD[] = { 0x06, 0x41, 0x69, 0xEF, 0x79, 0x94 };
const char* board = "RTLapseCAM";   //ESP Board Name

int avgLDR = 0;                     // Average value of the LDR sensor reading
int pc = 0;                         // For LDR
boolean frameCapture = false;
uint16_t frame_cnt = 0;
boolean recording = false;          //used to set recording to start.
boolean frameCaptured = false;      //when a frame is being captured
boolean idleTriggered = false;      //so idle trigger does not continue after stopped
long current_frame_time;
int frameSkip = 0;                  //Allows to skip every x frames (2 or 4) to speed up time-lapse
int frameSkipCnt = 0;               //Tracking counter
long time_to_wait_till_print_done = 120 * 1000;  //120 seconds. Need to allot for time for Z to move + time for exposure. Also consider when print is large and z has to move all the way up. Assumption is print is done if no transition from UV on to off for X seconds
long prevUpdateMillis = 0;
long updateIntervalMS = 2000;       //update the display status (bottom area) every 2 seconds
char battStats[40] = "" ; 


GoProControl gp(GOPRO_SSID, GOPRO_PASS, CAMERA, remote_MAC_ADD, board);
movingAvg LDRSenReading(4);         //setup the averaging object to use 4 values



void setup()
{
  pinMode(LDRSensorPIN, INPUT);
  //pinMode(36, INPUT);
  //gpio_pulldown_dis(GPIO_NUM_25);  //G36/25 share the same port, so set 25 as floating - https://github.com/m5stack/M5StickC-Plus/blob/master/README.md
  //gpio_pullup_dis(GPIO_NUM_25);

  M5.begin(); // Initialize M5StickC Plus.
  
  LDRSenReading.begin(); //moving average
  M5.Axp.EnableCoulombcounter();  //Enable Coulomb counter - battery monitoring M5Stick-C Plus

  //M5.Beep.beep();
 
  /*
  tone(BUZZER_PIN, NOTE_C4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  tone(BUZZER_PIN, NOTE_D4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  tone(BUZZER_PIN, NOTE_E4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  tone(BUZZER_PIN, NOTE_F4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  tone(BUZZER_PIN, NOTE_G4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  tone(BUZZER_PIN, NOTE_A4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  tone(BUZZER_PIN, NOTE_B4, 500, BUZZER_CHANNEL);
  noTone(BUZZER_PIN, BUZZER_CHANNEL);
  */
  
  if (debug) {gp.enableDebug(&Serial);}   //enable serial debugging output
  
  //xTaskCreate(keep_alive, "keep_alive", 10000, NULL, 1, NULL);  //optional for ESP32
  
  M5.Lcd.setRotation(3);  //Rotate the screen.
  setupHMI();

  updateStatus("    C O N N E C T I N G", 2, TFT_YELLOW);
  delay(2000);

  //M5.Lcd.setTextColor(TFT_RED,TFT_BLACK);    M5.Lcd.setTextFont(4);
  //M5.Lcd.setTextSize(3);  //Set font size.  

  // TESTING
  /*
  updateStatus("   C O N N E C T E D !", 2, TFT_GREEN);
  delay(2000);
  updateStatus("        S U C C E S S", 2, TFT_GREEN);
  delay(2000);
  updateStatus("          R E A D Y !", 2, TFT_GREEN);
  delay(2000);
  updateStatus("    Closing GP Comms", 2, TFT_RED);
  delay(2000);
  updateStatus("    Closing GP Comms", 2, TFT_RED);
  delay(2000);
  updateStatus("    Closing GP Comms", 2, TFT_RED);
  delay(2000);
  updateStatus("    Recording Started", 2, TFT_GREEN);
  delay(2000);
  updateStatus(" Idle Timeout Triggered", 2, TFT_YELLOW);
  delay(2000);
  */

  //Connect to the GoPro
  while (!gp.isConnected())
  {
    gp.begin();
  }

 
  //show connected state
  updateStatus("    C O N N E C T E D !", 2, TFT_GREEN);
  delay(1000);

  //set photo mode to ensure GoPro is in the right mode
  updateStatus("Set Photo Mode", 2, TFT_GREEN);
  gp.setMode(PHOTO_MODE);
  updateStatus("        S U C C E S S", 2, TFT_GREEN);
  delay(1000);
  updateStatus("           R E A D Y !", 2, TFT_GREEN);

  if (debug) {Serial.println(">>        Printing Status...");}
  if (debug) {gp.printStatus();}   //prints to serial monitor   
}
//#################################################################################################

void loop()
{
  unsigned long currUpdateMillis = millis();
  M5.update();    //Read the press state of the key.
  
  //regular updates to the HMI
  if (currUpdateMillis - prevUpdateMillis > updateIntervalMS)
  {
    updateStats(); 
    //Serial.println("Should only be every 2 seconds..");

    //Once recording, show other status updates like battery etc.
    if (recording)
    {
      updateStatus(formatM5BattInfo(M5.Axp.GetBatVoltage(), M5.Axp.GetBatPower(), M5.Axp.GetVBusVoltage(), battStats), 2, TFT_GREEN);
      //Serial.print("Batt Stuff:  ");Serial.println(formatM5BattInfo(M5.Axp.GetBatVoltage(), M5.Axp.GetBatPower(), M5.Axp.GetVBusVoltage(), battStats));  
    }
    
    
    prevUpdateMillis = currUpdateMillis;
  }
    
  //Read and average the photoresistor readings
  pc = analogRead(LDRSensorPIN); // read the photocell
  avgLDR = LDRSenReading.reading(pc);    // calculate the moving average
  //Serial.print("LDR: ");Serial.println(avgLDR);

 

  //if (M5.BtnA.wasPressed())     //This is the main button on top "M5"
  //if (M5.BtnA.pressedFor(2000))  //Button on side.  Use for hard reset - pressed for 2 seconds or more.
  
  if (M5.BtnB.wasPressed())
  {
    //if (debug) {Serial.println("Closing GoPro connection.");}
    //updateStatus("    Closing GP Comms", 2, TFT_RED);
    //M5.Beep.beep();
    //gp.end();   //close GoPro connection
    //delay(2000);
    //esp_restart(); 
    if (debug) {Serial.println("Change SkipFrame.");}
    changeFrameSkip();
    delay(2000);
  }

    
  if (M5.BtnA.pressedFor(2000))  //pressed for 2 seconds or more.  change frame skip val
  {
    updateStatus("    Closing GP Comms", 2, TFT_RED);
    M5.Beep.beep();
    gp.end();   //close GoPro connection
    delay(2000);    
  }

  if (M5.BtnA.wasPressed())    //start/stop recording
  {
     recording = !recording;  //toggle the value
     //M5.Beep.beep();
     updateStatus("    Recording Started", 2, TFT_GREEN);

     if (recording && !gp.isConnected())  //if not connected and recording started, try to re-connect
     {
       gp.begin();
     }

     if (!recording)  //if recording was stopped, also close the connection with the GoPro (IMPORTANT!)
     {
      updateStatus("   Recording Stopped", 2, TFT_YELLOW);
      idleTriggered = false;  //reset
      frame_cnt=0;            //reset
      gp.end();
     }
     
     if (debug) {Serial.print("Recording toggled as: ");Serial.println(recording);}
     current_frame_time = millis();   //reset idle trigger time when recording starts or stops
  }

  //Idle Timeout:  use a timer to set recording false if idle for a period of time
  if (millis() > current_frame_time + time_to_wait_till_print_done && !idleTriggered && recording)
  {
    recording = false;   //stop recording
    gp.end();
    if (debug) {Serial.println(" Idle Timeout triggered:  Stopping");}
    updateStatus(" Idle Timeout Triggered", 2, TFT_YELLOW);
    idleTriggered = true;
    M5.Beep.beep();M5.Beep.beep();
  }
  
  if (recording)
  {
    //Serial.println("Recording started");
    if (avgLDR > 1000 && !frameCaptured)  //only take snap when LDR triggered by UV light from the printer.  1 picture per layer
    {
      //take picture here
      if (gp.isConnected())
      {
        frameCaptured = true;    //ensures it doesnt trigger rapid sequences due to LDR staying on
        if (!CanISkipAFrame())   //skipping frames?
        {
          gp.shoot();
          if (debug) {Serial.println("Taking a picture");}
          current_frame_time = millis();   //resets the counter with each frame so Idle Timeout doesnt get called
          frame_cnt++;  
          //frameCaptured = true;
          delay(1000);
        }
      }
    
    }
    else if (avgLDR < 1000)  //avoids taking many pictures while the LDR values settle.  Only set it back when LDR is dark
    {
      frameCaptured = false;  
    }
  }
 

  gp.keepAlive(); // not needed on HERO3 but needed on later GP's
  delay(10);
}

boolean CanISkipAFrame()
{
  if (debug) {Serial.print("FrameSkipSetting: "); Serial.println(frameSkip);} 
  if (debug) {Serial.print("FrameSkipCount: "); Serial.println(frameSkipCnt);} 
  if (debug) {Serial.print("FrameCount: "); Serial.println(frame_cnt);} 
  if (debug) {Serial.println("");} 

  if (frameSkip > 0)    //we are skipping some frames
  {
    //when in 2 frame skip mode
    if (frameSkip == 2)
    {
      if(frameSkipCnt <= 1)
      {
        frameSkipCnt++;
        return true;    //still not at 2, skip
      } else {
        frameSkipCnt=0;
        return false;   //2nd frame, take a pic
      }
    }
    
    //when in 4 frame skip mode
    
    if (frameSkip == 4)
    {
      if(frameSkipCnt <= 3)
      {
        frameSkipCnt++;     //still not at 4, skip
        return true; 
      } else {
        frameSkipCnt=0;
        return false;       //4th frame, take a pic
      }
    }
  }
  return false;         //not skipping frames.  Record as normal
  
}
//######################################################################################

void changeFrameSkip()
{
  char msg[40] = "";
  //options are 0,2 ,4
  frameSkip = frameSkip + 2;         //increment by 2
  if (frameSkip > 4) frameSkip = 0;  //reset back to 0
  frameSkipCnt=0;       //reset
  sprintf(msg, "    FrameSkip now: %d", frameSkip);
  updateStatus(msg, 2, TFT_GREEN);
  if (debug) {Serial.print("FrameSkipVal:  ");Serial.print(frameSkip);}
}


//######################################################################################

const char *formatM5BattInfo(float battV, float battPwr, float usbV, char * battStats)
{
  //char buffer[40] = "";
  //char bV[6];
  //char bP[6];
  //char uV[6];

  //dtostrf(battV, 6, 1, bV);
  //dtostrf(battPwr, 6, 1, bP);
  //dtostrf(usbV, 6, 1, uV);

  //sprintf(buffer, "BattV: %2.1f V BattP: %2.1f V USBV: %2.1f V", bV, bP, uV);
  sprintf(battStats, "BV: %2.1fV BPw: %2.1fmW USBV: %2.1fV", battV,battPwr, usbV);
  //Serial.println(buffer);
  //Serial.print(battV);Serial.print("   ");
  //Serial.print(battPwr);Serial.print("   ");
  //Serial.println(usbV);
  
  return battStats;
  
  //sprintf(displayString, "value:%7d.%d%d", (int)num, int(num*10)%10, int(num*100)%10);
  //char buffer[40];
  //sprintf(buffer, "The %d burritos are %s degrees F", numBurritos, tempStr);
  //Serial.println(buffer);
  //dtostrf(float_value, min_width, num_digits_after_decimal, where_to_store_string)
}
//########################################################################################

void lcdClear()
{
  M5.Lcd.fillScreen(BLACK);  
}
//########################################################################################

void updateStats()
{

  M5.Lcd.setFreeFont(&Orbitron_ExtraBold9pt7b);
  
  //update COMMS  - TO DO

  //Update Record State
  M5.Lcd.fillRect(120, 45, 100, 20, TFT_BLACK);  //screen is 135H x 240W  //fillRect(x, y, w, h, color);
  M5.Lcd.setCursor(130, 60); 
  if (recording)
  {
    M5.Lcd.setTextColor(TFT_GREEN); 
    M5.Lcd.print("O N");   

  } else {
    M5.Lcd.setTextColor(TFT_RED); 
    M5.Lcd.print("O F F");      
  }
  //show frame skip status
  M5.Lcd.fillRect(185, 45, 55, 20, TFT_BLACK);
  M5.Lcd.setCursor(195, 60);
  M5.Lcd.setTextColor(TFT_GREEN);
  M5.Lcd.print("S:");  
  M5.Lcd.print(frameSkip);  
  
  //Update Frame Count
  M5.Lcd.fillRect(120, 65, 100, 20, TFT_BLACK);  //screen is 135H x 240W  //fillRect(x, y, w, h, color);
  M5.Lcd.setCursor(135, 80); 
  M5.Lcd.setTextColor(TFT_GREEN); 
  M5.Lcd.print(frame_cnt);   
}
//########################################################################################

void updateStatus(String textToShow, int xVal, uint16_t clr)
{
  //M5.Lcd.fillRect(2, 105, 238, 30, TFT_BLACK);  //screen is 135H x 240W  //fillRect(x, y, w, h, color);
  //M5.Lcd.drawRoundRect(4, 105, 232, 28, 4, TFT_PURPLE);
  //M5.Lcd.setCursor(xVal, 125); //x,y
  
  M5.Lcd.fillRect(0, 85, 240, 50, TFT_BLACK);  //screen is 135H x 240W  //fillRect(x, y, w, h, color);
  M5.Lcd.drawRoundRect(4, 85, 232, 48, 4, TFT_PURPLE);  //drawRoundRect(x, y, w, h, radius, colour)
  M5.Lcd.setCursor(xVal, 105); //x,y

  
  M5.Lcd.setTextColor(clr);
  M5.Lcd.println(textToShow);
}
//########################################################################################

void setupHMI()
{
  M5.Lcd.setTextColor(TFT_BLUE);
  M5.Lcd.setFreeFont(&Orbitron_ExtraBold14pt7b); 
  M5.Lcd.setCursor(10, 30); 
  M5.Lcd.println("RTLapseCAM");  

  //M5.Lcd.setFreeFont(&Orbitron_ExtraBold9pt7b); 
  //M5.Lcd.setCursor(2, 60);   //x y
  //M5.Lcd.setTextColor(TFT_BLUE); 
  //M5.Lcd.print("C O M M S:"); 

  M5.Lcd.setFreeFont(&Orbitron_ExtraBold9pt7b); 
  M5.Lcd.setCursor(2, 60); 
  M5.Lcd.setTextColor(TFT_BLUE); 
  M5.Lcd.print("R E C O R D"); 
  M5.Lcd.setCursor(140, 60); 
  M5.Lcd.setTextColor(TFT_RED); 
  M5.Lcd.print("O F F");    

  M5.Lcd.setCursor(2, 80); 
  M5.Lcd.setTextColor(TFT_BLUE); 
  M5.Lcd.print("F R A M E #"); 
  M5.Lcd.setCursor(140, 80); 
  M5.Lcd.setTextColor(TFT_GREEN); 
  M5.Lcd.print("0"); }

  
/*
void keep_alive(void *parameter)  //only for ESP32
{
  while (1)
  {
    gp.keepAlive();
  }
  vTaskDelete(NULL);
}
*/

/*
void doCommands(char* serIn)
{
  
  switch (serIn)
  {
  default:
    break;

  // Connect
  case 'b':
    gp.begin();
    break;

  case 'c':
    Serial.print("Connected: ");
    Serial.println(gp.isConnected() == true ? "Yes" : "No");
    break;

  case 'p':
    gp.confirmPairing();
    break;

  case 's':

    if (CAMERA == HERO3)
    {
      char * statusChar;
      statusChar = gp.getStatus();
      Serial.println("Status :");
      for(int i = 0; i < 56; i++)
      {
        Serial.print(statusChar[i], HEX);Serial.print(" ");
      }
      Serial.println("");
      Serial.println("End Status.");
      if (statusChar[0] == 0x00){Serial.println("camera ON");}
      else{Serial.println("camera OFF");}
      free(statusChar); // Don't forget to free memory
    }

    else
    {
      char * statusChar;
      statusChar = gp.getStatus();
      Serial.println("Status :");
      Serial.println(statusChar);
      free(statusChar); // Don't forget to free memory
    }
    
    break;

  case 'm': // DO NOT USE WHEN CAMERA IS OFF, IT FREEZE ESP
    char* medialist;
    medialist = gp.getMediaList();
    Serial.println("Media List:");
    Serial.println(medialist);
    free(medialist); // Don't forget to free memory
    break;

  // Turn on and off
  case 'T':
    gp.turnOn();
    break;

  case 't':
    gp.turnOff();
    break;

  // Take a picture of start a video
  case 'A':
    gp.shoot();
    break;

  // Stop the video
  case 'S':
    gp.stopShoot();
    break;

  // Check if it is recording
  case 'r':
    Serial.print("Recording: ");
    Serial.println(gp.isRecording() == true ? "Yes" : "No");
    break;

  // Set modes
  case 'V':
    gp.setMode(VIDEO_MODE);
    break;

  case 'P':
    gp.setMode(PHOTO_MODE);
    break;

  case 'M':
    gp.setMode(MULTISHOT_MODE);
    break;

  // Change the orientation
  case 'U':
    gp.setOrientation(ORIENTATION_UP);
    break;

  case 'D':
    gp.setOrientation(ORIENTATION_DOWN);
    break;

  // Change other parameters
  case 'f':
    gp.setVideoFov(MEDIUM_FOV);
    break;

  case 'F':
    gp.setFrameRate(FR_120);
    break;

  case 'R':
    gp.setVideoResolution(VR_1080p);
    break;

  case 'h':
    gp.setPhotoResolution(PR_12MP_WIDE);
    break;

  case 'L':
    gp.setTimeLapseInterval(60);
    break;

  // Localize the camera
  case 'O':
    gp.localizationOn();
    break;

  case 'o':
    gp.localizationOff();
    break;

  // Delete some files, be carefull!
  case 'l':
    gp.deleteLast();
    break;

  case 'g':
    gp.deleteAll();
    break;

  // Print useful data
  case 'd':
    gp.printStatus();
    break;

  // Close the connection
  case 'X':
    gp.end();
    break;
  }
}
*/
