
/*
Aug 2023 Copyright - www.PlastiBots.com
Usage: Source code, schematics and build information is provided to maintain the spirit of open source. 
Per that model, you are authorized to re-use, change, modify and update the code as long as original 
credits are maintained and published for public consumption. It may not be used for commercial, re-sale 
or other for-profit purposes unless written consent is provided.

If you find this project helpful, consider donating to the makers behind this work.

My project log: http://www.plastibots.com/index.php/2023/08/31/ninebot-g30lp-scooter-custom-3d-printed-led-side-markers-tail-light/

Credits and useful links:  
 https://www.tweaking4all.com/hardware/arduino/adruino-led-strip-effects/#LEDStripEffectColorWipe
 + also note links to relevant libraries below.
 OTA: 
 https://community.platformio.org/t/esp32-ota-using-platformio/15057/5
 https://github.com/espressif/arduino-esp32/blob/master/libraries/ArduinoOTA/examples/BasicOTA/BasicOTA.ino
*/

#include <Arduino.h>
#include <Adafruit_NeoPixel.h>
#include <Adafruit_Sensor.h>            //ADXL345 gyro / accell
#include <Adafruit_ADXL345_U.h>         //ADXL345 gyro / accell
#include <BH1750.h>                     //https://github.com/claws/BH1750
#include <Wire.h>                       //for BH1750 light sensor
#include <movingAvg.h>                  // https://github.com/JChristensen/movingAvg


//#####################################################################################
//#define debug                           //Be sure to comment out when done debugging!!
//#####################################################################################

//#####################################################################################
//Blynk can be used to monitor values for light etc.  To use it, create a blynk project with a terminal widget and buttons as noted in the blog
// Ensure this is commented (not defined) once in prod as Ninebot will be driven outside of range and crashes the Blynk connection when out of range
//#define useBlynk 
//#####################################################################################

//######################################################################################################################
unsigned long lightsOffDelay = 60000;     // timer to wait to turn lights back off - after LUX meets low threshold
int lightLUXthreshold = 65;               //values lower than this trigger lights on.  Aim to have lights on at dusk
int velocityThreshold = 1;                //value related to gyro X readings that trigger rear light blinking when higher than this
//######################################################################################################################

//OTA
#include <WiFi.h>
#include <ESPmDNS.h>
#include <WiFiUdp.h>
#include <ArduinoOTA.h>
//make sure ESP is on main WiFi not guest newtork - only need this if using Blynk
const char* ssid     = "SSID";
const char* password = "PASSWORD";
//const char* myHostname = "SegwayMax"; // Hostname
//END OTA


//must have these before including Blynk
#ifdef useBlynk
  #define BLYNK_TEMPLATE_ID "TMPL2upPW8u23"  //use whatever you created 
  #define BLYNK_TEMPLATE_NAME "ESP32Base"    //use whatever you created 
  #define BLYNK_AUTH_TOKEN "AUTHToken"
  //#define BLYNK_PRINT Serial
  #include <WiFiClient.h>
  #include <BlynkSimpleEsp32.h>
#endif


#define PIXELPIN_FRONTMARKER   D9       // set to pin connected to data input of WS8212 (NeoPixel) strip
#define NUMPIXELS_FRONTMARKER 10        // number of LEDs (NeoPixels) in your strip  End state will be 7
#define PIXELPIN_REARMARKER   D8        // set to pin connected to data input of WS8212 (NeoPixel) strip
#define NUMPIXELS_REARMARKER 9          // number of LEDs (NeoPixels) in your strip  End state will be 7
#define PIXELPINTAIL     D7             // set to pin connected to data input of WS8212 (NeoPixel) strip
#define NUMPIXELS_TAIL    5             // number of LEDs (NeoPixels) in your strip  End state will be 7

#define MARKER_BRIGHT_MAX    200        // LED brightness (1 to 255) – start with low values!
#define MARKER_BRIGHT_NORM   140        // max LED brightness (1 to 255)
#define MARKER_BRIGHT_LOW   40          // max LED brightness (1 to 255)

#define TAIL_BRIGHT_MAX    200          // max LED brightness (1 to 255)
#define TAIL_BRIGHT_NORM   100          // LED brightness (1 to 255) – start with low values!
#define TAIL_BRIGHT_LOW    20           // max LED brightness (1 to 255)
//#define GRAVITY          20           // Gravitational acceleration  - default 20                                        // Bounce coefficient at the lowest point = R1 / R2
//#define R1               3            // Bounce coefficient 1  default 3
//#define R2               5            // Bounce coefficient 2  default 5  >> 8 is nice
//#define REFMINV          -80          // Minimum bounce speed

// Colours   http://www.barth-dev.de/online/rgb565-color-picker/
//#define GLOW 0x1F51
//#define VERYDARKGREY 0x2904
//#define VERYVERYDARKGREY RGB565(10, 10, 10)

//#define GREEN 0x08e70c
//#define ORANGE 0x800000
//#define RED 0x100000
//#define BLUE 0x0000ff


int pos = 0, dir = 1;                      // For KITT:  Position, direction of "eye"
uint8_t col = 1;                           // Color of falling LED block

unsigned long avgAccellLastTime = 0;
unsigned long avgAccellTimeDelay = 10;    //sample for averaging every XX ms
unsigned long doBlinkTailLastTime = 0;
unsigned long doBlinkTailDelay = 2000;    // timer to blink tails when accel shows moving
unsigned long getLightReadingLastTime = 0;
unsigned long getLightReadingDelay = 200; // timer for taking light sensor readings
unsigned long checklightLastTime = 0;
unsigned long checklightDelay = 1000;     // timer for checking light
unsigned long lightsOffLastTime = 0;


//float lux = lightMeter.readLightLevel();
int lightSensorData = 0;                  //read the sensor
int lightSensorMovingAvg = 0;             //add data point and get moving average value
bool isMoving = false;                    //triggers by checking roll and pitch

int retryCount = 0;                       //# of times to retry connection
//float r, gyroX, gyroY, gyroZ;
float r;
int16_t ax, ay, az;
int roll, pitch;

int accellCounter=0;                      //acceleration averaging counter
boolean lightsOn = true;                  //used to determine when lights should be on or off
boolean doLightsLastState = false;        //track state
float SpeedFactor = 0.006;               // Breath sequence - changing this number changes the speed - default 0.0006 - seems to want to stay wiht 6's
float StepDelay = 5;                      // Breath sequence - ms for a step delay on the lights



#ifdef useBlynk
  bool sendGyrotoBlynk=1, sendLUXtoBlynk=1;
#endif

Adafruit_NeoPixel RearSidemarkerStrip = Adafruit_NeoPixel(NUMPIXELS_REARMARKER, PIXELPIN_REARMARKER, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel FrontSidemarkerStrip = Adafruit_NeoPixel(NUMPIXELS_FRONTMARKER, PIXELPIN_FRONTMARKER, NEO_GRB + NEO_KHZ800);
Adafruit_NeoPixel tailStrip = Adafruit_NeoPixel(NUMPIXELS_TAIL, PIXELPINTAIL, NEO_GRB + NEO_KHZ800);

/* Assign a unique ID to this sensor at the same time */
Adafruit_ADXL345_Unified accel = Adafruit_ADXL345_Unified(12345);  //Adafruit_ADXL345.ADXL345(address=0x1d, busnum=1)  ADXL345 is ox53

/* Setup the lightMeter*/
BH1750 lightSensor;  //0x23

/* Setup Moving Average*/
movingAvg lightAvg(20);

/* Setup Moving Average for accell values*/
//movingAvg accelX(20);
//movingAvg accelY(20);
//movingAvg accelZ(20);

// put function declarations here:
//int myFunction(int, int);
//void colorWipeMarker(byte red, byte green, byte blue, int SpeedDelay);
void doKITT(int dly, int howmany);
void doBounce(uint8_t BLOCK);
void doRearTailLightStartup();
void set_block(uint16_t y, uint8_t c, uint8_t blk);
void setAllMarker(byte red, byte green, byte blue);
void setAllTail(byte red, byte green, byte blue);

void colorFadeMarkers(uint8_t r, uint8_t g, uint8_t b, uint8_t wait);
void colorFadeTail(uint8_t r, uint8_t g, uint8_t b, uint8_t wait);
void setPixel(int Pixel, byte red, byte green, byte blue);
void setPixelTail(int Pixel, byte red, byte green, byte blue);
void setBrakesFlasher(int howMany, bool doEndSlowFlash);
void doTailRunningLight();
void sendToBlynkTerminal();
void RunningLights(byte red, byte green, byte blue, int WaveDelay);
void sidemarkerBreath();

void doOTA();

/* ADXL345 */
void displayRange(void);
void displayDataRate(void);
void displaySensorDetails(void);
float mapfloat(float x, float in_min, float in_max, float out_min, float out_max);
/* END ADXL345 */

#ifdef useBlynk
  //to get the value from the terminal text input on Web/Mobile App
  BLYNK_WRITE(V0) {
      String TerminalVal = param.asStr();
  }
  //From Blynk App - turn LUX  data on/off
  BLYNK_WRITE(V1) {
     sendLUXtoBlynk = param.asInt();
  }
  //From Blynk App - turn gyro data on/off
  BLYNK_WRITE(V2) {
    sendGyrotoBlynk = param.asInt();
  }
  //From Blynk App - light value threshold to flip between off on
  BLYNK_WRITE(V3) {
    lightLUXthreshold = param.asInt();
    Blynk.virtualWrite(V0, "lightLUXThreshold changed:"); 
    Blynk.virtualWrite(V0, lightLUXthreshold);
    Blynk.virtualWrite(V0, "\n");    
  }
  //From Blynk App - velocity threshold to determine rear tail blinking
  BLYNK_WRITE(V4) {
     velocityThreshold = param.asInt();
     Blynk.virtualWrite(V0, "velocityThreshold changeed:"); 
     Blynk.virtualWrite(V0, velocityThreshold);
     Blynk.virtualWrite(V0, "\n");  
  }
#endif

//#######################################################################################################


void setup() {
  pinMode(PIXELPIN_REARMARKER, OUTPUT);
  
  #ifdef debug
    Serial.begin(115200); 
  #endif

  //DONT USE WIRE: Already baked into Adafruit driver and how I2C works! both sensors using default XIAO I2C SDA SCL.  Assume it's already done in Adafruit ADXL345 init.
  //Wire.begin(D2,D3);  // XIAO SDA is D4, SCL D5 - using own pins
  //Wire.begin(D5,D4);  // XIAO SDA is D4, SCL D5 - if on same pins as ADXL345


  // initialize LED sidemarkerStrip
  RearSidemarkerStrip.begin();
  FrontSidemarkerStrip.begin();
  tailStrip.begin();

  RearSidemarkerStrip.show();
  FrontSidemarkerStrip.show();
  tailStrip.show();
  
  #ifdef useBlynk
    Blynk.virtualWrite(V0, "Showing startup sequences...\n"); 
  #endif

  setAllMarker(0, 0, 0);                               //all side marker LEDs off
  setAllTail(0, 0, 0);                                 //all tail LEDs off
  
  //for (int i=0; i<10; i++)
  //{
    //RunningLights(0,0,0xff, 50); 
  //  sidemarkerBreath(); 
  //}

  //setAllTail(0, 200, 0);
  //Blue for both first
  setAllTail(0, 0, TAIL_BRIGHT_MAX);
  colorFadeMarkers(0, 0, MARKER_BRIGHT_MAX, 25);
  
  //OTA - Do this after initial LED startup
  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, password);
  //try to connect, but will often not be within WiFi range so gracefully fail.
  while (WiFi.waitForConnectResult() != WL_CONNECTED) {
    #ifdef debug 
      Serial.println("Connection Failed! Rebooting..."); 
    #endif
    delay(1000);
    if (retryCount > 2) break;  //try a few times, then break out
    retryCount++;
  }
  
  //only if connected
  if (WiFi.status() == WL_CONNECTED)
  {
    doOTA();

    #ifdef useBlynk
      //Blynk.begin(BLYNK_AUTH_TOKEN, ssid, password, "blynk.cloud", 80);
      Blynk.config(BLYNK_AUTH_TOKEN, "blynk.cloud", 80);  //using wifi connection prior to blynk
      
      Blynk.virtualWrite(V0, "clr"); 
      Blynk.virtualWrite(V0, "Connected to Blynk...\n"); 
      Blynk.virtualWrite(V1, 1);                                        //update the app to reflect the hardware setting
      Blynk.virtualWrite(V2, 1);                                        //update the app to reflect the hardware setting
      Blynk.virtualWrite(V3, lightLUXthreshold);                        //update the app to reflect the hardware setting
      Blynk.virtualWrite(V4, velocityThreshold);                        //update the app to reflect the hardware setting
    #endif
  }
  //END OTA
    
  delay(5000);
  //Then over to red and orange
  colorFadeTail(TAIL_BRIGHT_NORM, 0, 0, 5);                       //Fade to red  
  colorFadeMarkers(MARKER_BRIGHT_NORM, 30, 0, 5);                 //Sidemarkers to orange
  
  #ifdef useBlynk
    Blynk.virtualWrite(V0, "Showing startup sequences... DONE\n"); 
  #endif 


  /* Initialise the ADXL345 sensor */
  if (!accel.begin())
  {
     #ifdef debug
       Serial.println("Ooops, no ADXL345 detected ... Check your wiring!");
    #endif
  }
  /* Set the range to whatever is appropriate for your project */
      //accel.setRange(ADXL345_RANGE_16_G);
      //  accel.setRange(ADXL345_RANGE_8_G);
      //  accel.setRange(ADXL345_RANGE_4_G);
  accel.setRange(ADXL345_RANGE_2_G);
  
  lightSensor.begin();              //star the sensor
  lightAvg.begin();                 //start moving average
  //accelX.begin();                   //start moving average for accell X
  //accelY.begin();                   //start moving average for accell Y
  //accelZ.begin();                   //start moving average for accell Z
  
  #ifdef useBlynk
    Blynk.virtualWrite(V0, "\nSetup complete...\n"); // send first log
  #endif



}
//#######################################################################################################

void loop() {

  if (WiFi.status() == WL_CONNECTED)  //only want to do this when at home
  {
    ArduinoOTA.handle();

    #ifdef useBlynk
      Blynk.run();
    #endif
  }
  

  //get accell readings and update moving averages for each axis
  sensors_event_t event;
  accel.getEvent(&event);
  
  //gyroX = event.acceleration.x;
  //gyroY = event.acceleration.y;
  //gyroZ = event.acceleration.z;
  ax = event.acceleration.x;
  ay = event.acceleration.y;
  az = event.acceleration.z;
  r = sqrt(ax*ax+ay*ay+az*az);
  roll   = 180/M_PI * ( M_PI/2 - (acos(ay/r) ) );
  pitch  = 180/M_PI * ( M_PI/2 - (acos(ax/r) ) );
  //totalMvmt = abs(gyroX) + abs(gyroY) + abs(gyroZ);
 
  if ((abs(roll) > velocityThreshold) || (abs(pitch) > velocityThreshold))
  {
    isMoving = true;
  } else {
    isMoving = false;
  }
 
  //only blink the rear tail light when moving forward and mode is lightsOn (accellerating or constant fwd motion) every x seconds
  if (((millis() - doBlinkTailLastTime) > doBlinkTailDelay) && (isMoving) && (lightsOn)) 
  {
    setBrakesFlasher(5, false);
    doTailRunningLight();
    doBlinkTailLastTime = millis();
    #ifdef debug
      Serial.print("<<TAIL BLINKING>>");
    #endif
    #ifdef useBlynk
      Blynk.virtualWrite(V0, "\nBlinking tails! \n"); // send first log
    #endif
  }


  // Build up average of light sensor reading values
  if ((millis() - getLightReadingLastTime) > getLightReadingDelay) 
  {  
    lightSensorData = lightSensor.readLightLevel();  //read the sensor
    lightSensorMovingAvg = lightAvg.reading(lightSensorData);  //add data point and get moving average value
    getLightReadingLastTime = millis();
  }

  // Check to see if lights should be turned on / off
  if ((millis() - checklightLastTime) > checklightDelay) 
  {
    if (lightSensorMovingAvg < lightLUXthreshold)           //getting dark, turn lights on
    {
      if (!lightsOn)                                        //only do if not already done
      {
        doTailRunningLight();                               //turn on tail light
        colorFadeMarkers(255, 30, 0, 5);                    //turn on sidemarkers 
        #ifdef debug
          Serial.print("<<Lights On>>");
        #endif
      }
      lightsOn = true;
      lightsOffLastTime = millis();
      //sidemarkerBreath(); 

    }else{    //getting light outside, turn lights donw low  - but want to wait for a period of time to avoid false positives (e.g. street/car lights)
      
      if (((millis() - lightsOffLastTime) > lightsOffDelay) && (lightsOn))
      {
        //setAllMarker(MARKER_BRIGHT_LOW, 0, 0);            //marker light low
        colorFadeMarkers(MARKER_BRIGHT_LOW, 5, 0, 5);       //Sidemarkers
        setAllTail(TAIL_BRIGHT_LOW, 0, 0);                  //tail light low
        
        #ifdef debug
           Serial.print("<<Lights Off>>");
        #endif
        lightsOn = false;
      }
    }
    checklightLastTime = millis();

    #ifdef debug
      Serial.print("X:");Serial.print(ax); Serial.print("  | Light: "); Serial.print(lightSensorMovingAvg); Serial.print(" lux LightsOn:");Serial.println(lightsOn);
    #endif


    #ifdef useBlynk
      sendToBlynkTerminal();
    #endif

  }
 
 //Now do this:  https://www.arduino.cc/reference/en/libraries/movingavg/

}
// END LOOP
//####################################################################################################################
void doOTA()
{

  ArduinoOTA
    //.setPort(8266); defaults to this already
    //.setHostname(myHostname)
    .onStart([]() {
      String type;
      if (ArduinoOTA.getCommand() == U_FLASH)
        type = "sketch";
      else // U_SPIFFS
        type = "filesystem";

      // NOTE: if updating SPIFFS this would be the place to unmount SPIFFS using SPIFFS.end()
      #ifdef debug 
        Serial.println("Start updating " + type); 
      #endif
    })
    .onEnd([]() {
      #ifdef debug 
        Serial.println("\nEnd");  
      #endif
    })
    .onProgress([](unsigned int progress, unsigned int total) {
      #ifdef debug 
        Serial.printf("Progress: %u%%\r", (progress / (total / 100)));
      #endif
    })
    .onError([](ota_error_t error) {
      #ifdef debug 
        Serial.printf("Error[%u]: ", error);
        if (error == OTA_AUTH_ERROR)    Serial.println("Auth Failed");
        else if (error == OTA_BEGIN_ERROR)  Serial.println("Begin Failed");
        else if (error == OTA_CONNECT_ERROR) Serial.println("Connect Failed");
        else if (error == OTA_RECEIVE_ERROR) Serial.println("Receive Failed");
        else if (error == OTA_END_ERROR) Serial.println("End Failed"); 
      #endif
    });

  ArduinoOTA.begin();

  #ifdef debug 
    Serial.println("Ready");Serial.print("IP address: ");Serial.println(WiFi.localIP()); 
  #endif

}
//####################################################################################################################
#ifdef useBlynk
  void sendToBlynkTerminal()
  {
    //for sending data to Blyk Terminal
    char terminalStr[50];
    char lightAvgChar[4]; 
    char acllXChar[5],acllYChar[5],acllZChar[5],isMovingChar[3], rollChar[15], pitchChar[15];
    char lightsOnChar[2];
    strcpy(terminalStr, ">");
      if (sendGyrotoBlynk)
      {
        sprintf(acllXChar, "%0.2f", ax);
        sprintf(acllYChar, "%0.2f", ay);
        sprintf(acllZChar, "%0.2f", az);
        sprintf(rollChar, "%d", roll);
        sprintf(pitchChar, "%d", pitch);
        sprintf(isMovingChar, "%i", isMoving);
        //sprintf(totalMvmtChar, "%0.2f", totalMvmt);
    
        //strcat(terminalStr, " X:");
        //strcat(terminalStr, acllXChar);
        //strcat(terminalStr, "  Y:");
        //strcat(terminalStr, acllYChar);
        //strcat(terminalStr, "  Z:");
        //strcat(terminalStr, acllZChar);
        //strcat(terminalStr, "  TTL:");
        //strcat(terminalStr, totalMvmtChar);     
        strcat(terminalStr, "  Rll:");
        strcat(terminalStr, rollChar);     
        strcat(terminalStr, "  Pit:");
        strcat(terminalStr, pitchChar);     
        strcat(terminalStr, "  Moving?:");
        strcat(terminalStr, isMovingChar);     
      }
      if (sendLUXtoBlynk) 
      {
        itoa (lightSensorMovingAvg, lightAvgChar, 10);
        itoa (lightsOn, lightsOnChar, 10);
        strcat(terminalStr, "  LUX: ");
        strcat(terminalStr, lightAvgChar);
        strcat(terminalStr, "  LightsOnYN: ");
        strcat(terminalStr, lightsOnChar);
      }
      strcat(terminalStr, "\n");
      Blynk.virtualWrite(V0, terminalStr);
      memset(terminalStr, 0, sizeof(terminalStr));
    }
#endif

/*
void colorWipeMarker(byte red, byte green, byte blue, int SpeedDelay) 
{
  for(uint16_t i=0; i<NUMPIXELS_FRONTMARKER; i++) {  //pixels
    for(uint16_t j=0; j<green; j+=10) {  //ramp up each pixel brightness
       setPixel(i, 0, j, 0);                    
       RearSidemarkerStrip.show();
       FrontSidemarkerStrip.show();
       delay(1);
    }
    RearSidemarkerStrip.show();
    FrontSidemarkerStrip.show();
    delay(SpeedDelay);
  }
}
*/
//####################################################################################################################
void doRearTailLightStartup() 
{
    for(uint16_t i=0; i<TAIL_BRIGHT_MAX; i++) {  //center pixel #3
       //setPixel(i, 0, j, 0);                    
       //strip.setPixelColor(Pixel, strip.Color(red, green, blue));
       tailStrip.setPixelColor(2, tailStrip.Color(i, 0, 0));
       tailStrip.show();
       delay(1);
    }
    for(uint16_t i=0; i<TAIL_BRIGHT_MAX; i++) {  //next 2 outer pixels  #1 and 3
       tailStrip.setPixelColor(1, tailStrip.Color(i, 0, 0));
       tailStrip.setPixelColor(3, tailStrip.Color(i, 0, 0));
       tailStrip.show();
       delay(5);
    }
    for(uint16_t i=0; i<TAIL_BRIGHT_MAX; i++) {  //next outer pixels #0 and 5
       tailStrip.setPixelColor(0, tailStrip.Color(i, 0, 0));
       tailStrip.setPixelColor(4, tailStrip.Color(i, 0, 0));
       tailStrip.show();
       delay(15);
    }
}
//####################################################################################################################

void doTailRunningLight()
{
      for(uint16_t i=0; i<NUMPIXELS_TAIL; i++) {  
       tailStrip.setPixelColor(i, tailStrip.Color(TAIL_BRIGHT_NORM, 0, 0));
      }
      tailStrip.show();
}
//####################################################################################################################


void setAllMarker(byte red, byte green, byte blue) 
//only going to do 9 pixels (based on rear) even though front has 10. 
//works out as bottom pixel at front is exposed outside the reflective tape and is bright.
{
  for (int i = 0; i < NUMPIXELS_FRONTMARKER; i++ ) 
  {
    RearSidemarkerStrip.setPixelColor(i, RearSidemarkerStrip.Color(red, green, blue));
    FrontSidemarkerStrip.setPixelColor(i, FrontSidemarkerStrip.Color(red, green, blue));
  }
  RearSidemarkerStrip.show(); 
  FrontSidemarkerStrip.show();
}
//####################################################################################################################

void setAllTail(byte red, byte green, byte blue) 
{
  for (int i = 0; i < NUMPIXELS_TAIL; i++ ) 
  {
    tailStrip.setPixelColor(i, tailStrip.Color(red, green, blue));
  }
  tailStrip.show(); 
}
//####################################################################################################################


//####################################################################################################################


void colorFadeTail(uint8_t r, uint8_t g, uint8_t b, uint8_t wait) 
{
  
      uint8_t startR, startG, startB;
      while ((startR != r) || (startG != g) || (startB != b))        // while the curr color is not yet the target color
      {  
        
        //if (debug){Serial.print("Insde ColorFade...");}
        uint32_t startColor = tailStrip.getPixelColor(0); // get the current colour
        startB = startColor & 0xFF;
        startG = (startColor >> 8) & 0xFF;
        startR = (startColor >> 16) & 0xFF;  // separate into RGB components
        
        //if (debug){Serial.print("Insde ColorFade...2");}
        if (startR < r) startR++; else if (startR > r) startR--;  // increment or decrement the old color values
        if (startG < g) startG++; else if (startG > g) startG--;
        if (startB < b) startB++; else if (startB > b) startB--;
        setAllTail(startR, startG, startB);  
        //if (debug){Serial.print("Insde ColorFade..4");}
                                   //strip.show();
        //delay(1);  //REMOVED WAS CAUSING ESP TO CRASH!!!! add a delay if its too fast
        //if (debug){Serial.print("Insde ColorFade..5");}
        delay(wait);
      }  
}
//####################################################################################################################

void colorFadeMarkers(uint8_t r, uint8_t g, uint8_t b, uint8_t wait) 
{
  
      uint8_t startR, startG, startB;
      while ((startR != r) || (startG != g) || (startB != b))        // while the curr color is not yet the target color
      {  
        
        //if (debug){Serial.print("Insde ColorFade...");}
        uint32_t startColor = RearSidemarkerStrip.getPixelColor(0); // get the current colour
        startB = startColor & 0xFF;
        startG = (startColor >> 8) & 0xFF;
        startR = (startColor >> 16) & 0xFF;  // separate into RGB components
        
        //if (debug){Serial.print("Insde ColorFade...2");}
        if (startR < r) startR++; else if (startR > r) startR--;  // increment or decrement the old color values
        if (startG < g) startG++; else if (startG > g) startG--;
        if (startB < b) startB++; else if (startB > b) startB--;
        //strip.setPixelColor(i, startR, startG, startB);  // set the color
        //strip.setPixelColor(i, 255, 0, 0);  // set the color
        //if (debug){Serial.print("Insde ColorFade..3");}
        setAllMarker(startR, startG, startB);  
        //if (debug){Serial.print("Insde ColorFade..4");}
                                   //strip.show();
        //delay(1);  //REMOVED WAS CAUSING ESP TO CRASH!!!! add a delay if its too fast
        //if (debug){Serial.print("Insde ColorFade..5");}
        delay(wait);
      }  
}
//####################################################################################################################

void setPixel(int Pixel, byte red, byte green, byte blue) 
{
  RearSidemarkerStrip.setPixelColor(Pixel, RearSidemarkerStrip.Color(red, green, blue));
  FrontSidemarkerStrip.setPixelColor(Pixel, FrontSidemarkerStrip.Color(red, green, blue));
}

//####################################################################################################################
void setPixelTail(int Pixel, byte red, byte green, byte blue) 
{
  tailStrip.setPixelColor(Pixel, tailStrip.Color(red, green, blue));
}

//####################################################################################################################


void setBrakesFlasher(int howMany, bool doEndSlowFlash) 
{
   //fast flash
    for(int i=0; i<howMany; i++)
    {
      setAllTail(0, 0, 0);
      delay(50);
      setAllTail(TAIL_BRIGHT_MAX, 0, 0);
      delay(50);
    }
    
    //finish with slow flash if enabled
    if (doEndSlowFlash)
    {
      for(int i=0; i<howMany; i++)
      {
        setAllTail(0, 0, 0);
        setAllTail(TAIL_BRIGHT_MAX, 0, 0);
        delay(250);
      }
     }
}
//####################################################################################################################

void setBrakes(byte red, byte green, byte blue, int SpeedDelay, int ReturnDelay) 
{
  
  for (int i = 0; i <= NUMPIXELS_TAIL; i++) 
  {
    setPixelTail(i, red, green, blue); 
    setPixelTail(i+1, red, green, blue);  
    setPixelTail(i+2, red-20, green, blue);  
    setPixelTail(i+3, red-40, green, blue);  
    setPixelTail(i+4, red-60, green, blue);  
    setPixelTail(i+5, red-80, green, blue);  
    tailStrip.show();
    delay(SpeedDelay);
  }
  delay(ReturnDelay);
}
//####################################################################################################################
void doKITT(int dly, int howmany)
{
   for (int i = 0; i <  NUMPIXELS_REARMARKER*howmany; i++) //each loop sweeps from one side to the other - do it a few times
   {
    
     int j;

     tailStrip.setPixelColor(pos - 2, 0x100000); // dark red 
     tailStrip.setPixelColor(pos - 1, 0x800000); // orange
     tailStrip.setPixelColor(pos    , 0xFF3000); // Center pixel is brightest
     tailStrip.setPixelColor(pos + 1, 0x800000); // orange 
     tailStrip.setPixelColor(pos + 2, 0x100000); // dark red 

     tailStrip.show();
     delay(dly);
 
     // Rather than being sneaky and erasing just the tail pixel,
     // it's easier to erase it all and draw a new one next time.
     for(j=-2; j<=2; j++) tailStrip.setPixelColor(pos+j, 0);  //count is based on the number of pixels set above
     // Bounce off ends of strip
     pos += dir;
     if(pos < 0) {
       pos = 1;
       dir = -dir;
     } else if(pos >= tailStrip.numPixels()) {
       pos = tailStrip.numPixels() - 3;
       dir = -dir;
     }
   }
   setAllTail(0, 0, 0);  //turn on the LED sweep at startup.
}
//####################################################################################################################


//####################################################################################################################
void set_block(uint16_t y, uint8_t c, uint8_t blk) 
{
  // LED block display
  //blk is number of pixels that forms each block
  // y Start position of LED block to be displayed
  // c Display color 0 (off), 1 (red) -8 (green) -16 (blue) -24 (red)
  uint32_t rgb;
  //Serial.print("Val of c="); Serial.println(c); 
  if (c == 0) rgb = tailStrip.Color (0,0,0);
  //else if (c <= 8) rgb = LED.Color (150,0,0); //DJA - to make it all red
  //else if (c <= 8) rgb = strip.Color ((uint16_t) (16-c) * BRIGHT >> 3,0,0); //DJA - to make it all red
  else if (c <= 8) rgb = tailStrip.Color (MARKER_BRIGHT_MAX,0,0); //DJA - to make it all red
  else return;

  for (uint8_t i = 0; i < blk; i++) 
  {
    if (y<0) break;
    //Serial.print("y="); Serial.println(y);
    tailStrip.setPixelColor ((NUMPIXELS_REARMARKER-1)-y, rgb);  //DJA - added for right side sweeping to center
    tailStrip.setPixelColor (y--, rgb);   //default
    //delay(30);
  }
}
//####################################################################################################################
void RunningLights(byte red, byte green, byte blue, int WaveDelay) {
  int Position=0;
 
  for(int j=0; j<NUMPIXELS_REARMARKER*2; j++)
  {
      Position++; // = 0; //Position + Rate;
      for(int i=0; i<NUMPIXELS_REARMARKER; i++) {
        // sine wave, 3 offset waves make a rainbow!
        //float level = sin(i+Position) * 127 + 128;
        //setPixel(i,level,0,0);
        //float level = sin(i+Position) * 127 + 128;
        setPixel(i,((sin(i+Position) * 127 + 128)/255)*red,
                   ((sin(i+Position) * 127 + 128)/255)*green,
                   ((sin(i+Position) * 127 + 128)/255)*blue);
      }
     
      RearSidemarkerStrip.show();
      delay(WaveDelay);
  }
}


//####################################################################################################################
void sidemarkerBreath()
{
  // Make the lights breathe
  for (int i = 0; i < 1024; i++) //65535
  {
    // Intensity will go from 10 - MaximumBrightness in a "breathing" manner
    float intensity = MARKER_BRIGHT_MAX /2.0 * (1.0 + sin(SpeedFactor * i));
    RearSidemarkerStrip.setBrightness(intensity);
    FrontSidemarkerStrip.setBrightness(intensity);
    //Now set every LED to that color
     for (int ledNumber=0; ledNumber<NUMPIXELS_FRONTMARKER; ledNumber++) 
     {
       RearSidemarkerStrip.setPixelColor(ledNumber, MARKER_BRIGHT_NORM, 30, 0);
       FrontSidemarkerStrip.setPixelColor(ledNumber, MARKER_BRIGHT_NORM, 30, 0);
     }
    RearSidemarkerStrip.show();
    FrontSidemarkerStrip.show();
    //Serial.println(i);
  }
}

